import Joi from 'joi';
import express from 'express';
import FormData from 'form-data';

import scanUrl from '../../service/scan-url';
import { permission } from '../../middlewares/permission'; 
import virustotal from '../../service/get-virus-totalkey';
import { sanitize } from '../../middlewares/sanitizer';
import { checkCompanyValidityStrategy, jwtStrategy } from '../../middlewares/strategy';
import { validateBody } from '../../middlewares/validator';
import { htmlsanitize } from '../../middlewares/sanitizeHtml';
import restrictFeature from '../../middlewares/featureBlocker';

let validator = Joi.object().keys({
    urlIds: Joi.array().items(
        Joi.string().hex().length(24).required()
    ).required(),
    reportedMailId: Joi.string().hex().length(24).required()
});


/**
 * @api {post} /api/v1/scan/scan-url scan-url
 * @apiName scan-url
 * @apiGroup scan
 * @apiDescription Scan a url
 * 
 * @apiVersion 1.0.0
 * 
 * @apiBody   {String}   urlId        		 urlId of the reported mail
 * @apiBody   {String}   reportedMailId      Id of the reported mail
 * 
 * @apiSuccess {Boolean} Success             true
 * @apiSuccess {String}  Message             Scan now in processing, Result Will be Updated Soon!
 * 
 * @apiSuccessExample Success-Response:
 *      HTTP/1.1 200 OK 
 * {
 *           "Success": true,
 *           "Message": "Scan now in processing, Result Will be Updated Soon!"
 * }
 * 
 * @apiError {Boolean} Success             false
 * @apiError {Array}   messages            "\"reportedMailId\" is not allowed to be empty"
 * 
 * @apiSuccessExample Error-Response:
 *       HTTP/1.1 400 Not Found
 * 
 *      "success":   false,
 *      "messages" : "\"reportedMailId\" is not allowed to be empty"
 * 
 */

let controller = async (req, res, next) => {
	try {
		const { reportedMailId, urlIds } = req.body; 
		const companyId = req.user.companyId;
		let apikey = await virustotal.getApiKey(companyId);

		for (const urlId of urlIds) {
			const url = await db.Url.findOne({
				_id: urlId, reportedMailId
			});
			if (!url) {
				throw new RequestError('Invalid urlId');
			}

			let formData = new FormData();
			formData.append('url', url.url);
			formData.append('apikey', apikey);
			scanUrl.processScanURL(formData, urlId, companyId);
			url.status = 'IN PROGRESS';
			await url.save();
		}

		return res.status(200).send({
			success: true,
			Message: "Scans are now in processing, results will be updated soon!"
		});
	} catch (error) {
		console.log(error);
		next(error);
	}
};

const apiRouter = express.Router();
apiRouter.route('/').post(htmlsanitize(), validateBody(validator), jwtStrategy, restrictFeature("TPIR-AALP"), checkCompanyValidityStrategy, permission('Emails','Write'), controller);
export default apiRouter;