import express from 'express';

import { jwtStrategy } from '../../middlewares/strategy';
import { permission } from '../../middlewares/permission';
import { htmlsanitize } from '../../middlewares/sanitizeHtml';
import restrictFeature from '../../middlewares/featureBlocker';

/**
 * @api {delete} /api/settings/smtp Delete SMTP Configuration
 * @apiName DeleteSMTPConfig
 * @apiGroup Settings
 * @apiPermission Settings Write
 * 
 * @apiHeader {String} Authorization Bearer token
 * 
 * @apiSuccess {Boolean} Success Indicates if the operation was successful
 * @apiSuccess {String} message Success message
 * 
 * @apiSuccessExample {json} Success-Response:
 *     HTTP/1.1 200 OK
 *     {
 *       "Success": true,
 *       "message": "SMTP Configuration Deleted Successfully!"
 *     }
 * 
 * @apiError (Error 400) RequestError Error message
 * @apiError (Error 500) ServerError Unexpected error
 */
const controller = async (req, res, next) => {
    try {
        const companyId = req.user.companyId;
        console.log("Deleting SMTP configuration for companyId", companyId);

        // SMTP fields to be deleted (marked with deletedAt)
        const smtpFields = [
            'smtpHost',
            'smtpPort',
            'smtpPassword',
            'smtpUsername',
            'smtpFrom',
            'smtpTo',
            'mailSmtp'
        ];

        // Set deletedAt for all SMTP related metadata
        const pendingQueries = smtpFields.map(name => {
            return db.MetaData.findOneAndUpdate(
                { name, companyId },
                { deletedAt: new Date() },
                { new: true }
            );
        });

        // Execute all update queries
        await Promise.all(pendingQueries);

        // Set mailSmtp to false
        await db.MetaData.findOneAndUpdate(
            { name: "mailSmtp", companyId },
            { value: false, deletedAt: null },
            { upsert: true, new: true, setDefaultsOnInsert: true }
        );

        return res.status(200).send({ Success: true, message: 'SMTP Configuration Deleted Successfully!' });
    } catch (error) {
        console.log('error', error);
        next(error);
    }
};

const apiRouter = express.Router();
apiRouter.route('/').post(
    htmlsanitize(),
    jwtStrategy,
    permission(''),
    controller
);

export default apiRouter;
