import Joi from 'joi';
import express from 'express';

import crypto from '../../crypto';
import { sanitize } from '../../middlewares/sanitizer';
import { jwtStrategy } from '../../middlewares/strategy';
import { validateBody } from '../../middlewares/validator';
import { permission } from '../../middlewares/permission';
import { htmlsanitize } from '../../middlewares/sanitizeHtml';

const validator = Joi.object().keys({
    apikey: Joi.string().required()
});

/**
 * @api {post} /api/v1/settings/virus-total virus-total
 * @apiName virus-total
 * @apiGroup Settings
 * @apiDescription Upload Virus Total key
 * 
 * @apiVersion 1.0.0
 * 
 * @apiBody   {String}   apikey        apiKey of virus total
 * 
 * @apiSuccess {Boolean} Success       true
 * @apiSuccess {String}  Message       Virus-total apikey updated successfully
 * 
 * @apiSuccessExample Success-Response:
 *      HTTP/1.1 200 OK 
 * {
 *           "Success": true,
 *           "Message": "Virus-total apikey updated successfully"
 * }
 * 
 * @apiError {Boolean} Success             false
 * @apiError {Array}   messages            "\"apikey\" is not allowed to be empty"
 * 
 * @apiSuccessExample Error-Response:
 *       HTTP/1.1 400 Not Found
 *       {
 *           "success": false,
 *           "messages": [
 *               "\"apikey\" is not allowed to be empty"
 *           ]
 *       }
 * 
 */

const controller = async (req, res, next) => {
    try {
        const { apikey } = req.body;
        const companyId = req.user.companyId;

        let metadata = await db.MetaData.findOne({
            companyId, name: 'virus-total'
        })
        if(metadata){
            await db.MetaData.updateOne({
                _id: metadata._id
            }, {
                value: await crypto.encrypt(apikey)
            })
        } else {
            await db.MetaData.create({
                companyId,
                name: 'virus-total',
                value: await crypto.encrypt(apikey)
            })
        }
        return res.status(200).send({ Success: true, message: 'Virus-total apikey updated successfully' })
    } catch (error) {
        console.log('error', error);
        next(error)
    }
}

const apiRouter = express.Router();
apiRouter.route('/').post(htmlsanitize(), jwtStrategy, permission('Settings','Write'), validateBody(validator), controller);
export default apiRouter;