import express from 'express';
import Joi from 'joi';

import { permission } from '../../middlewares/permission';
import { jwtStrategy, checkCompanyValidityStrategy } from '../../middlewares/strategy';
import { validateBody } from '../../middlewares/validator';
import { htmlsanitize } from '../../middlewares/sanitizeHtml';

// Validator: only status boolean
const validator = Joi.object().keys({
	status: Joi.boolean().required()
});

/**
 * @api {post} /api/v1/smartBanners/enable-disable-banner enable-disable-banner
 * @apiName enable-disable-banner
 * @apiGroup smartBanners
 * @apiDescription Enable / Disable company banner message meta (BannerMessage)
 *
 * @apiVersion 1.0.0
 *
 * @apiBody  {Boolean} status  true to enable (create or restore), false to disable (soft delete)
 *
 * @apiSuccess {Boolean} Success true
 * @apiSuccess {String}  message Action result
 *
 * @apiSuccessExample Success-Response (Enable):
 *  HTTP/1.1 200 OK
 *  {
 *    "Success": true,
 *    "message": "banner enabled"
 *  }
 *
 * @apiSuccessExample Success-Response (Disable):
 *  HTTP/1.1 200 OK
 *  {
 *    "Success": true,
 *    "message": "banner disabled"
 *  }
 */

const DEFAULT_BANNER_MESSAGE = 'Unusual activity has been detected in your email account, possibly indicating phishing, spoofing, or unauthorized access. Please review the flagged emails and avoid clicking unknown links or downloading suspicious attachments.';

const controller = async (req, res, next) => {
	try {
		const { status } = req.body; 
		const companyId = req.user.companyId;

		if (status) {
			const domains = await db.Domain.find({ companyId, deletedAt: null });
			if (!domains.length) {
				return res.status(400).send({ success: false, message: 'No domains found for company.' });
			}
			const hasCredentialledDomain = domains.some(d => d.isCredential);
			if (!hasCredentialledDomain) {
				return res.status(400).send({ success: false, message: 'At least one domain must have credentials configured before enabling banner.' });
			}
			// Enable: find existing (soft deleted or active)
			let meta = await db.MetaData.findOne({ companyId, name: 'BannerMessage' });
			if (meta) {
				if (meta.deletedAt) {
					await db.MetaData.updateOne({ _id: meta._id }, { $set: { deletedAt: null, value: meta.value || DEFAULT_BANNER_MESSAGE } });
				}
			} else {
				await db.MetaData.create({
					companyId,
					name: 'BannerMessage',
					value: DEFAULT_BANNER_MESSAGE
				});

			}
            console.log("Banner status: onnn", status);
			return res.status(200).send({ success: true, message: 'Threshold Auto-Banner Enabled' });
		} else {
			// Disable: set deletedAt = now if exists and not already
			await db.MetaData.updateOne({ companyId, name: 'BannerMessage', deletedAt: null }, { $set: { deletedAt: new Date() } });
            console.log("Banner status: on off", status);
			return res.status(200).send({ success: true, message: 'Threshold Auto-Banner Disabled' });
		}
	} catch (error) {
		next(error);
	}
};

const apiRouter = express.Router();
apiRouter.route('/')
	.post(htmlsanitize(), jwtStrategy, checkCompanyValidityStrategy, permission('Settings','Write'), validateBody(validator), controller);

export default apiRouter;

