import express from 'express';
import config from '../../config/index.js';
import crypto from '../../crypto.js';
import getToken from '../../service/get-outlook-token.js';

const controller = async (req, res, next) => {
    const redirectUrl = `${config.app.url}/settings?smtp`;
    try {
        const { tenant: tenantId, admin_consent, state: encodedMetaData } = req.query;
        console.log('Received query parameters:', { tenantId, admin_consent, encodedMetaData });
        if (admin_consent?.toLowerCase() !== 'true') {
            throw new RequestError('Admin consent was not granted.');
        }

        if (!tenantId || !encodedMetaData) {
            throw new RequestError('Invalid callback request. Required parameters are missing.');
        }

        const decodedData = JSON.parse(Buffer.from(encodedMetaData, 'base64').toString('utf-8'));
        const { companyId, senderEmail } = decodedData;
        console.log('Decoded metadata:', { companyId, senderEmail });
        if (!companyId) {
            throw new RequestError('Invalid state data. Company information is missing.');
        }

        const customSmtp = await db.MetaData.findOne({ name: 'mailSmtp', companyId });
        const isCustomSMTPEnabled = customSmtp?.value === true || customSmtp?.value === 'true';
        if (isCustomSMTPEnabled) {
            throw new RequestError('You have already configured custom SMTP.', 409);
        }

        const existingCred = await db.SmtpCred.findOne({ companyId, deletedAt: null });
        if (existingCred) {
            throw new RequestError('SMTP has already been configured for this company.');
        }

        console.log('No existing SMTP configuration found for companyId:', companyId);
        const clientId = config.azure.defaultMailAppClientId;
        const clientSecret = config.azure.defaultMailAppClientSecret;
        const tokenResponse = await getToken.getOutlookAuthToken(tenantId, clientId, clientSecret);
        if (!tokenResponse || !tokenResponse.access_token) {
            throw new RequestError('Invalid credentials - Unable to authorize with Microsoft Azure.');
        }
        console.log('Successfully obtained access token from Azure.', tokenResponse);
        await db.SmtpCred.create({
            companyId,
            tenantId: crypto.encrypt(tenantId),
            senderEmail,
            type: 'Oauth',
        });

        return res.redirect(`${redirectUrl}&success=true&message=Successfully authorized the application.`);

    } catch (error) {
        console.error('Error during Azure consent callback:', error);
        const errorMessage = error.message || 'An internal server error occurred. Please try again later.';
        return res.redirect(`${redirectUrl}&success=false&message=${encodeURIComponent(errorMessage)}`);
    }
};

const apiRouter = express.Router();
apiRouter.route('/').get(controller);

export default apiRouter;
