import express from 'express';
import { jwtStrategy } from '../../middlewares/strategy.js';
import { permission } from '../../middlewares/permission.js';
import config from '../../config';

/**
 * @api {get} /api/smtp/azure-consent. Generate Azure Consent URL
 * @apiName GenerateAzureConsentUrl
 * @apiGroup SMTP
 * @apiPermission admin
 *
 * @apiHeader {String} Authorization Bearer token.
 *
 * @apiSuccess {Boolean} success True.
 * @apiSuccess {String} message Success message.
 * @apiSuccess {Object} data Response data.
 * @apiSuccess {String} data.consentUrl The generated consent URL.
 *
 * @apiError (Conflict 409) {String} message Error message for existing configuration.
 * @apiError (Internal Server Error 500) {String} message Error message for missing client ID.
 */
const controller = async (req, res, next) => {
    const { companyId } = req.user;
    const { senderEmail } = req.body
    console.log('Received senderEmail:', senderEmail);
    try {
        const existingCred = await db.SmtpCred.findOne({ companyId, type: 'Oauth', deletedAt: null });
        if (existingCred) {
            throw new RequestError('You have already configured the SMTP with Oauth.');
        }
        console.log('No existing Oauth configuration found for companyId:', companyId);

        const defaultClientId = config.azure.defaultMailAppClientId;
        if (!defaultClientId) {
            throw new RequestError("Default Azure mailer app's client ID is not set in environment variables.");
        }
        
        console.log('Using default client ID:', defaultClientId);

        const encodedData = Buffer.from(JSON.stringify({ companyId: companyId.toString(), senderEmail })).toString('base64');
        const tpirRedirectUrl = `${config.azure.callbackDomain}/api/v1/smtp/azure-callback`;
        const generatedConsentUrl = config.azure.consentUrl
            .replaceAll('{{clientId}}', defaultClientId)
            .replaceAll('{{endcodedRedirectUrl}}', encodeURIComponent(tpirRedirectUrl))
            .replaceAll('{{encodedMetaData}}', encodedData)
            .replaceAll('{{hinted_domain}}', senderEmail.split('@')[1] ?? '');

        console.log('Generated Consent URL:', generatedConsentUrl);

        return res.status(200).json({
            success: true,
            message: 'Redirecting to Microsoft page for authorization.',
            data: { consentUrl: generatedConsentUrl },
        });
    } catch (error) {
        next(error);
    }
};

const apiRouter = express.Router();
apiRouter.route('/').post(jwtStrategy, permission(''), controller);

export default apiRouter;