import express from 'express';
import { jwtStrategy } from '../../middlewares/strategy.js';
import { permission } from '../../middlewares/permission.js';
import crypto from '../../crypto.js';

/**
 * @api {get} /api/smtp/get-smtp Get SMTP Configurations
 * @apiName GetSmtpConfigurations
 * @apiGroup SMTP
 * @apiPermission admin
 * @apiDescription Get all SMTP configurations for a company with their status.
 *
 * @apiHeader {String} Authorization Bearer token.
 *
 * @apiSuccess {Boolean} success True.
 * @apiSuccess {Array} data Array of SMTP configurations.
 * @apiSuccess {Object} data.oauthConfig OAuth configuration details (if exists).
 * @apiSuccess {Object} data.manualConfig Manual credentials configuration details (if exists).
 *
 * @apiError (Not Found 404) {String} message Error message when no configurations exist.
 */
const controller = async (req, res, next) => {
    try {
        const { companyId } = req.user;
        const response = {};

        // Get OAuth configuration
        const oauthConfig = await db.SmtpCred.findOne({ 
            companyId, 
            type: 'Oauth', 
            deletedAt: null 
        });

        if (oauthConfig) {
            response.oauthConfig = {
                id: oauthConfig._id,
                type: oauthConfig.type,
                authorized: !!oauthConfig.tenantId,
                senderEmail: oauthConfig.senderEmail,
            };
        }

        // Get Manual Credentials configuration
        const manualConfig = await db.SmtpCred.findOne({ 
            companyId, 
            type: 'Creds', 
            deletedAt: null 
        });

        if (manualConfig) {
            response.manualConfig = {
                id: manualConfig._id,
                type: manualConfig.type,
                senderEmail: manualConfig.senderEmail,
                tenantId: crypto.decrypt(manualConfig.tenantId),
                clientId: crypto.decrypt(manualConfig.clientId),
                secretId: crypto.decrypt(manualConfig.secretId),
                createdAt: manualConfig.createdAt,
                updatedAt: manualConfig.updatedAt
            };
        }

        if (Object.keys(response).length === 0) {
            return res.status(200).json({
                success: true,
                message: 'No SMTP configurations found',
                data: {}
            });
        }
        console.log('SMTP configurations retrieved successfully:', response);
        return res.status(200).json({
            success: true,
            data: response
        });

    } catch (error) {
        next(error);
    }
};

const apiRouter = express.Router();
apiRouter.route('/').get(jwtStrategy, permission(''), controller);

export default apiRouter;
