import Joi from 'joi';
import express from 'express';

import crypto from '../../crypto.js';
import { permission } from '../../middlewares/permission.js';
import { jwtStrategy } from '../../middlewares/strategy.js';
import { validateBody } from '../../middlewares/validator.js';
import getToken from '../../service/get-outlook-token.js';

const validator = Joi.object().keys({
    tenantId: Joi.string().required(),
    clientId: Joi.string().required(),
    clientSecret: Joi.string().required(),
    senderEmail: Joi.string().email().required(),
});

/**
 * @api {post} /api/smtp/manual-creds Add Manual SMTP Credentials
 * @apiName AddManualSmtpCredentials
 * @apiGroup SMTP
 * @apiDescription Add manual SMTP credentials for a company.
 *
 * @apiHeader {String} Authorization Bearer token.
 *
 * @apiBody {String} tenantId      The Azure tenant ID.
 * @apiBody {String} clientId      The Azure application client ID.
 * @apiBody {String} clientSecret  The Azure application client secret.
 * @apiBody {String} senderEmail   The email address to be used as the sender.
 *
 * @apiSuccess {Boolean} success  True.
 * @apiSuccess {String}  message  Success message.
 *
 * @apiError (Bad Request 400) {String} message Error message for invalid credentials.
 * @apiError (Conflict 409) {String} message Error message for existing configuration.
 */
const controller = async (req, res, next) => {
    try {
        const { clientId, clientSecret, tenantId, senderEmail } = req.body;
        const { companyId } = req.user;

        const customSmtp = await db.MetaData.findOne({ name: 'mailSmtp', companyId });
        const isCustomSMTPEnabled = customSmtp?.value === true || customSmtp?.value === 'true';
        if (isCustomSMTPEnabled) {
            throw new RequestError('You have already configured custom SMTP.', 409);
        }

        const existingCred = await db.SmtpCred.findOne({ companyId, deletedAt: null });
        if (existingCred) {
            throw new RequestError('You have already configured SMTP.', 409);
        }
        console.log('No existing SMTP configuration found for companyId:', companyId);
        try {
            const accessToken = await getToken.getOutlookAuthToken(tenantId, clientId, clientSecret);
            if (!accessToken || !accessToken.access_token) {
                throw new RequestError('Invalid Credentials - Unable to authorize with Microsoft Azure.');
            }
        } catch (error) {
            throw new RequestError('Invalid Credentials - Unable to authorize with Microsoft Azure.');
        }
        await db.SmtpCred.create({
            companyId,
            clientId: crypto.encrypt(clientId),
            secretId: crypto.encrypt(clientSecret),
            tenantId: crypto.encrypt(tenantId),
            senderEmail,
            type: 'Creds',
        });

        return res.status(200).json({ success: true, message: 'Credentials saved successfully.' });

    } catch (err) {
        next(err);
    }
};

const apiRouter = express.Router();
apiRouter.route('/').post(jwtStrategy, permission(''), validateBody(validator), controller);

export default apiRouter;
