import Joi, { custom } from 'joi';
import express from 'express';
import path from 'path'
import multer from 'multer';
import fs from 'fs'
const FileType = require('file-type');
const sharp = require('sharp');


import { checkCompanyValidityStrategy, jwtStrategy } from '../../middlewares/strategy';
import { permission } from '../../middlewares/permission';
import { validateBody } from '../../middlewares/validator';
import { file } from 'googleapis/build/src/apis/file';
import { scanFile } from '../../middlewares/scanAndUploadFile';
import restrictFeature from '../../middlewares/featureBlocker';

let templateLogoPath = path.join(path.dirname(require.main.filename), 'public', 'templateLogo');
const ALLOWED_MIME_TYPES = ['image/png', 'image/jpeg'];
const ALLOWED_EXTENSIONS = ['.png', '.jpg', '.jpeg'];

let storage = multer.diskStorage({
    destination: function (req, file, cb) {
        cb(null, templateLogoPath);
    },
    filename: function (req, file, cb) {
        const templateName = req.body.templateName.replace(/\s/g, '');

        const ts = Date.now();
        const fileName = `${req.user.companyId}_${templateName}_${ts}${path.extname(file.originalname)}`.trim();
        cb(null,fileName);
    }
})

const fileFilter = (req, file, cb) => {
    if (ALLOWED_MIME_TYPES.includes(file.mimetype)) {
        cb(null, true);
    } else {
        cb(new RequestError('Only PNG or JPG image files are allowed'), false);
    }
};

let uploadAttachment = multer({ storage: storage, limits: { fileSize: 10485760 }, fileFilter: fileFilter })

const validateRealFileType = async (filePath) => {
  const fileType = await FileType.fromFile(filePath);
  if (!fileType || !ALLOWED_MIME_TYPES.includes(fileType.mime)) {
    throw new RequestError('Uploaded file is not a valid PNG or JPG image');
  }
};


const controller = async (req, res, next) => {
    try {
        const { templateName, headerContent = '' ,bodyContent = '', footerContent = '' ,bgColor ='', setDefault = false} = req.body
        let companyId = req.user.companyId;
        const logo = req.file ? req.file.filename : null;


        if (req.file) {
            const ext = path.extname(req.file.originalname).toLowerCase();
            if (!ALLOWED_EXTENSIONS.includes(ext)) {
                throw new RequestError('Only PNG or JPG file extensions allowed');
            }

            await validateRealFileType(req.file.path);

            const buffer = await sharp(req.file.path).rotate().toFormat(ext === '.png' ? 'png' : 'jpeg').toBuffer();
            await fs.promises.writeFile(req.file.path, buffer);
        }

        let defaultTemplate = await db.Template.findOne({name: templateName })
        if(!defaultTemplate) throw new RequestError('Default Template Not Found!');

        if(setDefault == 'true') {
            const template = await db.Template.findOne({name:templateName, companyId})
            if(template && template.logo !== 'Tpirlogo.png' && template.logo !== null && template.logo !== 'null'){
                const filePath = path.join(appRoot, 'public', 'templateLogo', template.logo);
                fs.unlink(filePath, (err) => {
                    if (err) {
                      console.error('Error deleting the Logo:', err);
                    } else {
                      console.log('Logo deleted successfully.');
                    }
                  });
            } 
            else if(template && (template.logo == null || template.logo == 'null')) {
                await db.Template.deleteOne({name:templateName, companyId})
                return res.status(200).json({ 
                    success: true,
                    message:'Default Template Set Successfully!'
                })   
            }
            else if(!template){
                return res.status(200).json({ 
                    success: true,
                    message:'Default template already set!'
                })  
            }
            
            await db.Template.deleteOne({name:templateName, companyId})
            return res.status(200).json({ 
                success: true,
                message:'Default Template Set Successfully!'
            })   
        }
       
        const customTemplate = await db.Template.findOne({name: templateName, companyId})
        if(customTemplate){
            customTemplate.name = templateName;
            customTemplate.headerContent = headerContent
            customTemplate.bodyContent = bodyContent;
            customTemplate.footerContent = footerContent;
            customTemplate.bgColor = bgColor;
            customTemplate.logo = logo!=null ? logo : (req.body.logo == null || req.body.logo =='null') ? null : req.body.logo;
            await customTemplate.save()
        }
        else{
            await db.Template.create({name: templateName, description: defaultTemplate.description, headerContent, bodyContent, footerContent, bgColor, companyId, logo: logo!=null ? logo : defaultTemplate.logo })
        }


        res.status(200).json({ 
            success: true,
            message:'Template Updated Successfully!'
        })

    } catch (err) {
        console.log(err);
        next(err)
    }
}
const apiRouter = express.Router();
apiRouter.route('/').post(jwtStrategy, restrictFeature("TPIR-CET"), checkCompanyValidityStrategy, uploadAttachment.single('logo'), scanFile, controller);
export default apiRouter;