import Joi from 'joi';
import express from 'express';

import { permission } from '../../middlewares/permission';
import processUser from '../../service/process-user';
import credentials from '../../service/credentials';
import { checkCompanyValidityStrategy, checkLicenseStrategy, checkTaConnectionStrategy, jwtStrategy } from '../../middlewares/strategy';
import { validateBody } from '../../middlewares/validator';
import { htmlsanitize } from '../../middlewares/sanitizeHtml';
import { updateSyncingStatusOfGroups } from '../../service/update-syncing-status-groups';

const validator = Joi.object().keys({
    domainId: Joi.string().hex().length(24).required(),
    groups: Joi.array().allow(null)

});

/**
 * @api {get} /api/v1/users/sync-user?domainId=63b676c744ed3b8a723865971 sync-user
 * @apiName sync-user
 * @apiGroup users
 * @apiDescription Syncing Of domain's users
 * 
 * @apiVersion 1.0.0
 * 
 * @apiQuery   {String}   domainId      id of the domain whose users has to be synced
 * 
 * @apiSuccess {Boolean} success             true
 * @apiSuccess {Array}   messages            "syncing start successfully"
 * 
 * 
 * @apiSuccessExample Success-Response:
 *      HTTP/1.1 200 OK 
 * 
 *      {
 *          "success": true,
 *          "message": "syncing start successfully"
 *      }
 * 
 * @apiError (400) {Boolean} success-1             false
 * @apiError (400) {Array}   messages-1            "invalid domain recieved"
 * 
 * @apiErrorExample Error-Response:
 *       HTTP/1.1 400 Not Found
 *    {
 *      "success":   false,
 *      "messages" : "invalid domain recieved"
 *    }
 * 
 * @apiError (400) {Boolean} success-2             false
 * @apiError (400) {Array}   messages-2            "Domain Credentials not uploaded properly please upload and try again.""
 * 
 * @apiErrorExample Error-Response:
 *       HTTP/1.1 400 Not Found
 *      {
 *      "success":   false,
 *      "messages" : "Domain Credentials not uploaded properly please upload and try again."
 *      }
 * 
 * @apiError (400) {Boolean} success             false
 * @apiError (400) {Array}   messages            "Domain type not define"
 * 
 * @apiErrorExample Error-Response:
 *       HTTP/1.1 400 Not Found
 *      {
 *      "success":   false,
 *      "messages" : "Domain type not define"
 *      }
 * 
 */


const controller = async (req, res, next) => {
    const { domainId, groups } = req.body;
    console.log("-------groups-------",groups)
    const companyId = req.user.companyId;
    try {

        const isTacEnabled = await db.Company.findOne({_id: companyId}).isTacEnabled
        if(isTacEnabled) throw new RequestError("The users can be synced via TA!")

        var domain = await db.Domain.findOne({
            _id: domainId, companyId, deletedAt: null
        })

        if(!domain) throw new RequestError("Domain does not exist!")

        if(domain && domain.isCredential) var creds = await credentials.getCredentials(companyId, domainId);
        const isExchange = domain.domainTypeForOutlook === 'EXCHANGE';
        
        if(groups){ 
            for(let i=0;i<groups.length;i++){
                const groupName   = isExchange ? `${groups[i].label}-Exchange` : groups[i].label;
                let group = await db.Group.findOne({groupName, groupId: groups[i].value, companyId, domainId, deletedAt: null})
                if(!group) await db.Group.create({...groups[i], companyId, domainId, groupName, type: "Admin", groupId: groups[i].value})
            }
        }

        if(!domain){
            throw new RequestError('invalid domain recieved');
        }
        if(domain.domainType == 'Gsuite' || domain.domainType == 'Outlook'){
            if(domain.isCredential){
                // if(domain.isSyncingUser) return next(new RequestError(`Syncing Already in Progress for ${domain.domainName}. Kindly wait until syncing is completed!`))
                
                await db.Domain.updateOne({
                    _id: domainId
                }, {
                    isSyncingUser: true,
                    syncStartAt: new Date()
                })


                const isAnyGroupInSyncingState = await db.Group.find({isSyncingUsers: true, companyId, deletedAt: null})
                if(isAnyGroupInSyncingState.length) throw new RequestError("Syncing already in progress kindly wait until syncing is complete!")
                            
                let subDomainGroups = await db.SubDomainGroup.find({ domainId: domain._id, companyId },{ _id: 1, groupName: 1 })
                subDomainGroups = subDomainGroups?.map(group=>group.groupName)
                
                await db.Group.updateOne({groupName: "all-users-tpir", isDefaultGroup: true, companyId, deletedAt:null},{$set: {isSyncingUsers: true}})
                let p;
                // if(groups && groups.length > 0){
                //     p  = groups.map(async (group)=>{
                //     let grp = await db.Group.findOne({groupName: group.label, groupId: groups.value, companyId, deletedAt: null})
                //     await db.Group.updateOne({_id: grp._id},{$set: { isSyncingUsers: true }})    
                //     return processUser.process(companyId, domainId, domain.domainName, subDomainGroups, domain.domainType, false, domain.domainTypeForOutlook=='EXCHANGE', grp, creds);
                // })
                // console.log('-------------PPPPPP',p)
                // let res = await Promise.allSettled(p)
                // }
                // else {
                //     let group = await db.Group.findOne({groupName:"all-users-tpir", companyId, isDefaultGroup: true})
                //     await processUser.process(companyId, domainId, domain.domainName, subDomainGroups, domain.domainType, false, domain.domainTypeForOutlook=='EXCHANGE', group, creds)
                // }

                if (groups && groups.length > 0) {
                    const p = groups.map(async (group) => {
                      try {
                        const groupName   = isExchange ? `${group.label}-Exchange` : group.label;
                        const grp = await db.Group.findOne({
                          groupName,
                          groupId: group.value,
                          companyId,
                          deletedAt: null
                        });
                  
                        if (!grp) throw new Error(`Group not found: ${groupName}`);
                  
                        // Mark as syncing
                        await db.Group.updateOne({ _id: grp._id }, { $set: { isSyncingUsers: true } });
                  
                        // Process the group
                        await processUser.process(
                            companyId,
                            domainId,
                            domain.domainName,
                            subDomainGroups,
                            domain.domainType,
                            false,
                            domain.domainTypeForOutlook === 'EXCHANGE',
                            grp,
                            creds,
                            false, // isMirrorSync
                            null,  // allUsersGroupId
                            new Date(), // syncTs
                            0 // pageIndex
                          )
                  
                        // Success
                        return { status: 'fulfilled', groupId: grp._id };
                  
                      } catch (err) {
                        console.error(`Error processing group ${groupName}:`, err);
                        await db.Group.updateMany({companyId, isSyncingUsers: true},{$set:{isSyncingUsers: false}})
                        // Fallback update if group is partially fetched
                        await db.Group.updateOne(
                          { groupId: group.value, companyId },
                          { $set: { isSyncingUsers: false } }
                        );
                  
                        // Let it be marked as rejected
                        throw err;
                      }
                    });
                  
                    const results = await Promise.allSettled(p);
                  
                  
                } else {
                    // Fallback to default group
                    const group = await db.Group.findOne({
                      groupName: "all-users-tpir",
                      companyId,
                      isDefaultGroup: true
                    });
                  
                    if (group) {
                      await processUser.process(
                        companyId,
                        domainId,
                        domain.domainName,
                        subDomainGroups,
                        domain.domainType,
                        false,
                        domain.domainTypeForOutlook === 'EXCHANGE',
                        group,
                        creds,
                        false, // isMirrorSync
                        null,  // allUsersGroupId
                        new Date(), // syncTs
                        0 // pageIndex
                      );
                      await db.Domain.updateOne(
                        { _id: domainId },
                        { lastAllUsersSyncAt: new Date() }
                      );
                    } else {
                      console.warn("⚠️ Default group not found for company:", companyId);
                    }
                  }
                  

            } else {
                throw new RequestError('Domain Credentials not uploaded properly please upload and try again.');    
            }
        } else {
            throw new RequestError('Domain type not define');
        }
        return res.status(200).send({ Success: true, message: 'syncing start successfully' })
    } catch (error) {
            await db.Group.updateMany({companyId, isSyncingUsers: true},{$set:{isSyncingUsers: false}})
            await db.Domain.updateOne({
                _id: domainId
            }, {
                isSyncingUser: false,
                syncStartAt: new Date()
            })

        next(error)
    }
}

const apiRouter = express.Router();
apiRouter.route('/').post(htmlsanitize(), jwtStrategy, 
checkTaConnectionStrategy,
checkCompanyValidityStrategy, 
checkLicenseStrategy,
permission('Users','Write'),  validateBody(validator), controller);

export default apiRouter;