import fs from "fs";
import path from "path";
import multer from "multer";
import csv from "csv-parser";
import express from "express";
import { queue } from '../../kue';
import { scanFile } from "../../middlewares/scanAndUploadFile";
import { checkCompanyValidityStrategy, checkTaConnectionStrategy, jwtStrategy } from "../../middlewares/strategy";
import { permission } from "../../middlewares/permission";

const csvPathDir = path.join(path.dirname(require.main.filename), "Content", "csv");
const storage = multer.diskStorage({
  destination: (req, file, cb) => {
    cb(null, csvPathDir);
  },
  filename: (req, file, cb) => {
    cb(null, file.originalname);
  },
});

function determineDomainType(domain) {
    if (domain.domainType === "Outlook") {
      return domain.domainTypeForOutlook === "EXCHANGE" ? "Exchange" : "Outlook";
    }
    return domain.domainType === "Gsuite" ? "Gsuite" : "";
  }

const upload = multer({ storage });
const BATCH_SIZE = 1000; // Define your batch size here

// Function to validate a CSV row
function checkForValidRow(row) {
  return (
    row &&
    row.email &&
    row.email.includes("@") &&
    row.name &&
    /^[a-zA-Z ]+$/.test(row.name)
  );
}

// Function to create batches of users
function createBatches(users) {
  const batches = [];
  for (let i = 0; i < users.length; i += BATCH_SIZE) {
    batches.push(users.slice(i, i + BATCH_SIZE));
  }
  return batches;
}

// Process rows to create jobs for deactivation and activation
const processRows = async (rows, domainType, domain, companyId, groupId) => {
let addUsers = [];
  
  var groups = await db.SubDomainGroup.find({ domainId: domain._id });
  groups = groups.map((group)=>group.groupName)

  for (const row of rows) {
    if (
     (row.email.match(/@(.+)$/)[1].trim().toLowerCase() === domain.domainName || row.email.match(/([^@]+)@(?:[^.]+\.)?(.+\..+)/)[2].trim().toLowerCase() === domain.domainName || groups.includes(row.email.split("@")[1].toLowerCase()))) {
      addUsers.push(row);
    }
  };

  const addUsersBatches = createBatches(addUsers)   
  let addUsersBatchesLength = addUsersBatches.length
  if(addUsersBatches.length){
    await db.Group.updateOne({_id: groupId, companyId, deletedAt:null},{$set:{isSyncingUsers: true}})
    await db.Group.updateOne({groupName: "all-users-tpir", isDefaultGroup: true, companyId, deletedAt:null},{$set:{isSyncingUsers: true}})

      addUsersBatches.forEach(batch => {
        addUsersBatchesLength--
        queue.create('sync-users-csv', { users: batch, domain, domainType, companyId, groupId, isLast: addUsersBatchesLength == 0 ? true : false }).removeOnComplete(true).priority('high').save();
      });
  }
  else{
    let isAnyUserUploadedInCurrentGroup = await db.User.countDocuments({companyId, groups:{$in:[groupId]}})
    if(!isAnyUserUploadedInCurrentGroup) await db.Group.deleteOne({ _id: groupId, isDefaultGroup:false, companyId})
  }

};


// Controller function to handle CSV upload and processing
const controller = async (req, res, next) => {
  const { domainId, groupId } = req.body;
  const companyId = req.user.companyId;
  try {

    const domain = await db.Domain.findOne({ _id: domainId, companyId, deletedAt: null });
    if (!domain) throw new Error("Domain does not exist!");

    const group = await db.Group.findOne({_id:groupId, companyId, deletedAt: null})
    if (!group) throw new Error("Group does not exist!");

    const isAnyGroupInSyncingState = await db.Group.find({isSyncingUsers: true, companyId, deletedAt: null})
    if(isAnyGroupInSyncingState.length) throw new RequestError("Syncing already in progress kindly wait until syncing is complete!")

    var countOfUsersInGroup = await db.User.countDocuments({companyId, groups:{$in:[groupId]}, deletedAt:null})

    const domainType = determineDomainType(domain);

    const file = req.file;
    const csvReadStream = fs.createReadStream(file.path);
    const rows = [];

    const userLimit = (await db.MetaData.findOne({ companyId, name: "userLimit"})).value;
    const userLimitMessenger = (await db.MetaData.findOne({ companyId, name: "userLimitMessenger"})).value;

    const addedUsersCount = await db.User.countDocuments({companyId, deletedAt: null})
    const remainingLimit = Math.max(0, parseInt(userLimit)+parseInt(userLimitMessenger) - addedUsersCount)
    let usersCountInCsv = 0
    console.log({addedUsersCount,remainingLimit})
    
    let showExceededUsersInCsvThanRemaningLimitError = false 
    let showAccessError = {email: false, messenger: false}

    csvReadStream.pipe(csv())
    .on("headers", async (headers) => {
      const requiredHeaders = ["name", "email", "phoneNumber"];
      const extraHeaders = headers.filter((header) => !requiredHeaders.includes(header));
      const missingHeaders = requiredHeaders.filter((header) => !headers.includes(header));
      if (missingHeaders.length > 0) {
        csvReadStream.destroy();
        next(new RequestError(`Missing required columns: ${missingHeaders.join(", ")}`))
      }
      if (extraHeaders.length > 0) {
        csvReadStream.destroy();
        if(!countOfUsersInGroup) await db.Group.deleteOne({_id: groupId, companyId, isDefaultGroup: false})
        next(new RequestError(`Unexpected columns found: ${extraHeaders.join(", ")}. Only "name", "email", and "phoneNumber" are allowed.`))
      }
    })
    .on("data", async (row) => {
      console.log({usersCountInCsv, remainingLimit, row})
      usersCountInCsv++;
      if(usersCountInCsv > remainingLimit) {
        // csvReadStream.destroy();  // Stop reading the file
        // next(new RequestError("The number of users in the CSV exceeds your maximum user license limit. Please remove some users from your CSV or upgrade your license.",429));
        showExceededUsersInCsvThanRemaningLimitError = true
      }

      if (checkForValidRow(row)) {
        rows.push({ name: row.name, email: row.email.trim().toLowerCase(), phoneNumber: row.phoneNumber });
      }
    }).on("end", async () => {
        try {

            fs.unlink(file.path, (err) => {
              if (err) console.error("Error deleting the uploaded CSV file:", err);
            });

            const allUsersGroup = await db.Group.findOne({groupName: "all-users-tpir", isDefaultGroup: true, companyId, deletedAt:null})
            
            await processRows(rows, domainType, domain, companyId, groupId);
                        
            const checkAccessLimit = async (group, accessType, limit, errorKey) => {
              if (group[accessType]) {
                const userCount = await db.User.countDocuments({
                  companyId,
                  deletedAt: null,
                  groups: { $in: [group._id] },
                });
                if (userCount >= limit) showAccessError[errorKey] = true;
              }
            };
            
            await Promise.all([
              checkAccessLimit(group, "emailAccess", userLimit, "email"),
              checkAccessLimit(group, "messengerAccess", userLimitMessenger, "messenger"),
              checkAccessLimit(allUsersGroup, "emailAccess", userLimit, "email"),
              checkAccessLimit(allUsersGroup, "messengerAccess", userLimitMessenger, "messenger"),
            ]);

            res.status(200).json({ message: "CSV file successfully processed.", showExceededUsersInCsvThanRemaningLimitError, showAccessError });
        
        } catch (error) {
          if(!countOfUsersInGroup) await db.Group.deleteOne({_id: groupId, companyId, isDefaultGroup: false})
            next(new RequestError(error))
        }
    });
    
  } catch (error) {
    console.error(error);
    if(!countOfUsersInGroup)await db.Group.deleteOne({_id: groupId, companyId, isDefaultGroup: false})
    next(new RequestError(error));
  }
};

// Function to determine domain type
const apiRouter = express.Router();
apiRouter.route("/").post(
    jwtStrategy,
    checkCompanyValidityStrategy,
    checkTaConnectionStrategy,
    permission("Users", "Write"),
    upload.single("csvFile"),
    scanFile,
    controller
  );
export default apiRouter;
