import Joi from 'joi';
import express from 'express';

import { permission } from '../../middlewares/permission';
import { checkCompanyValidityStrategy, jwtStrategy } from '../../middlewares/strategy';
import { validateBody } from '../../middlewares/validator';
import { htmlsanitize } from '../../middlewares/sanitizeHtml';

const validator = Joi.object().keys({
    messengerStatus: Joi.boolean().required(),
    userId: Joi.string().hex().length(24).required()
});


/**
 * @api {post} /api/v1/users/user-enable-disable user-enable-disable
 * @apiName user-enable-disable
 * @apiGroup users
 * @apiDescription Enabling or disabling a particular user's mailbox check
 * 
 * @apiVersion 1.0.0
 * 
 * @apiBody   {Boolean}   status      status of the user
 * @apiBody   {String}    userId      id of users has to be synced
 * 
 * @apiSuccess {Boolean}  Status      true
 * @apiSuccess {Array}    Message     "status changed successfully"
 * 
 * 
 * @apiSuccessExample Success-Response:
 *      HTTP/1.1 200 OK 
 * 
 *      {
 *          "Status": true,
 *          "Message": "status changed successfully"
 *      }
 * 
 * @apiError (400) {Boolean} success             false
 * @apiError (400) {Array}   messages            "Invalid user recieved"
 * 
 * @apiErrorExample Error-Response:
 *       HTTP/1.1 400 Not Found
 *      {
 *      "success":   false,
 *      "messages" : "Invalid user recieved"
 *      }
 * 
 */


const controller = async (req, res, next) => {
    try {
        const { userId, messengerStatus } = req.body;
        let user = await db.User.findById(userId)

        if(!user){
            throw new RequestError('Invalid user recieved');
        }
        if(!user.groups.length) throw new RequestError('User does not belong to any group with messenger access in company!')

        if(messengerStatus){
            let checkGroupsOfCompanyWithMessengerAccess = await db.Group.find({companyId: user.companyId, deletedAt:null, messengerAccess:true}).select('_id')
            checkGroupsOfCompanyWithMessengerAccess = checkGroupsOfCompanyWithMessengerAccess.map(group=> group._id.toString())
            let userGroups = user.groups.map(group=> group._id.toString())
            
            if(userGroups.some(group => checkGroupsOfCompanyWithMessengerAccess.includes(group))){
                await db.User.updateOne({
                    _id: user._id
                }, {
                    isActiveMessenger: messengerStatus
                })
                return res.status(200).send({ Success: true, message: 'status changed successfully' })
            }
            else throw new RequestError('User does not belong to any group with messenger access in company!')
        }
        await db.User.updateOne({
            _id: user._id
        }, {
            isActiveMessenger: messengerStatus
        })
        return res.status(200).send({ Success: true, message: 'status changed successfully' })
    } catch (error) {
        console.log('error', error);
        next(error)
    }
}

const apiRouter = express.Router();
apiRouter.route('/').post(htmlsanitize(), jwtStrategy, checkCompanyValidityStrategy, permission('Users','Write'), validateBody(validator), controller);
export default apiRouter;