import fs from 'fs';
import Joi from 'joi';
import path from 'path';
import express from 'express';
import { parseAsync } from 'json2csv';

import config from '../../config';
import { jwtStrategy } from '../../middlewares/strategy';
import { validateBody } from '../../middlewares/validator';
import { permission } from '../../middlewares/permission'

const validator = Joi.object().keys({
    domainId: Joi.string().hex().length(24).optional()
});

const fields = ['name', 'email', 'userId', 'status'];
const opts = { fields };

/**
 * @api {get} /api/v1/users/user-list-csv?domainId=63b676c744ed3b8a723865971 user-list-csv
 * @apiName user-list-csv
 * @apiGroup users
 * @apiDescription downloading user's list csv file of a particular domain or all-users-tpir
 * 
 * @apiVersion 1.0.0
 * 
 * @apiQuery   {Boolean}   [domainId]   id of the domain 
 * 
 * @apiSuccess {Boolean}  Success       true
 * @apiSuccess {String}   fileName      name of the csv file
 * @apiSuccess {String}   url           url of the csv file
 * 
 * 
 * @apiSuccessExample Success-Response:
 *      HTTP/1.1 200 OK 
 * 
 *   {
 *       "Success": true,
 *       "fileName": "1675067639919.csv",
 *       "url": "https://new-stag.threatalertbutton.com/static/attachments/"
 *   }
 * 
 */


const controller = async (req, res, next) => {
    try {
        const { domainId } = req.query;
        const companyId = req.user.companyId;
        const condition = { companyId, deletedAt: null };
        if(domainId) condition['domainId'] = domainId;
        const users = await db.User.find(condition)
        let data = [];
        for(let i=0; i<users.length; i++){
            data.push({
                name: users[i].name,
                email: users[i].email,
                userId: users[i].userId,
                status: users[i].isActive?'Active': 'InActive',
            })
        }
        parseAsync(data, opts)
        .then(csv => {
            let fileName = `${Date.now()}.csv`
            let filePath = path.join(appRoot, 'public', 'attachments', fileName);
            fs.writeFile(filePath, csv, err => {
                if (err) {
                    throw new RequestError(err);
                }
                return res.status(200).send({ Success: true, fileName, url: `${config.app.url}/api/static/attachments/` })
            });
        })
        .catch(err => {
            throw new RequestError(err);
        });
    } catch (error) {
        console.log('error', error);
        next(error)
    }
}

const apiRouter = express.Router();
apiRouter.route('/').get(jwtStrategy, validateBody(validator), permission('Users','Write'), controller);
export default apiRouter;