import Joi from 'joi';
import express from 'express';

import { jwtStrategy } from '../../middlewares/strategy';
import { validateBody } from '../../middlewares/validator';
import { permission } from '../../middlewares/permission';

const validator = Joi.object().keys({
    page: Joi.number().min(0).required(),
    limit: Joi.number().min(1).required(),
    domains: Joi.array().optional(),
    group: Joi.string().optional(),
    subDomainGroups: Joi.array().optional(),
    search: Joi.string().optional().allow(null).empty(''),
    status: Joi.string().optional().allow(null).empty(''),
    isActiveEmail: Joi.string().optional(),
    isActiveWhatsapp: Joi.string().optional(),
});

/**
 * @api {post} /api/v1/users/user-list?page=1&limit=10 user-list
 * @apiName user-list
 * @apiGroup users
 * @apiDescription Fetch the list of all-users-tpir
 * 
 * @apiVersion 1.0.0
 * 
 * @apiQuery   {Number}   page          (Pagination) Required page of the events records
 * @apiQuery   {Number}   limit         (Pagination) Required limit of event records to be returned
 * @apiQuery   {String}   domainId      Id of the domain who users are to be fetched
 * @apiQuery   {String}   [search]      Search for a particulat user
 * 
 * @apiSuccess {String} Success             true
 * @apiSuccess {Array}  users               list of all-users-tpir
 * @apiSuccess {Number} count               count of users
 * 
 * @apiSuccessExample Success-Response:
 *      HTTP/1.1 200 OK 
 * {
 *           "Success": true,
 *           "users": [
 *               {
 *                   "_id": "63d0efab032f943fc31e4ebe",
 *                   "name": "admintab",
 *                   "email": "admintab@kdmarc.com",
 *                   "userId": "193848ac-8363-482d-8578-2574b51c7d7d",
 *                   "domainId": "63d0eaaa032f943fc31e4ea2",
 *                   "companyId": "63d0ea61032f943fc31e4e80",
 *                   "isActive": true,
 *                   "emailType": "Outlook",
 *                   "deletedAt": null,
 *                   "__v": 0,
 *                   "createdAt": "2023-01-25T09:00:27.105Z",
 *                   "updatedAt": "2023-01-25T09:00:27.105Z"
 *               }
 *               
 *           ],
 *           "count": 1
 *           }
 * 
 * @apiError {Boolean} Success             false
 * @apiError {Array}   messages            "\"reportedMailId\" is not allowed to be empty"
 * 
 * @apiSuccessExample Error-Response:
 *       HTTP/1.1 400 Not Found
 * 
 *      "success":   false,
 *      "messages" : "\"reportedMailId\" is not allowed to be empty"
 * 
 */

const controller = async (req, res, next) => {
    try {
        const { page, limit, search = '', isActiveEmail, isActiveWhatsapp } = req.query;
        const companyId = req.user.companyId;
        let condition = { companyId, deletedAt: null };

        if(search && search.length){
            const regex = new RegExp(search, "i");
            condition['$or'] = [
                { name: regex }, 
                { email: regex }
            ]
        }
        // Add isActive and isActiveMessenger filters to the main condition if provided
        if (isActiveEmail === 'active') {
            condition.isActive = true;
        } else if (isActiveEmail === 'inactive') {
            condition.isActive = false;
        }
        if (isActiveWhatsapp === 'active') {
            condition.isActiveMessenger = true;
        } else if (isActiveWhatsapp === 'inactive') {
            condition.isActiveMessenger = false;
        }
        let count = await db.User.countDocuments(condition)
        let users = await db.User.aggregate([
            {
                $match: condition
            },
            {
                $lookup: {
                    from: "groups", 
                    localField: "groups",
                    foreignField: "_id",
                    as: "userGroups"
                }
            },
            {
                $addFields: {
                    belongToAGroupWithEmailAccess: {
                        $anyElementTrue: "$userGroups.emailAccess"
                    },
                    belongToAGroupWithMessengerAccess: {
                        $anyElementTrue: "$userGroups.messengerAccess"
                    }
                }
            },
            { $skip: (page - 1) * limit },
            { $limit: parseInt(limit) },
        ]);
        
        let groupsWithEmailAccess = await db.Group.find({companyId, emailAccess: true, deletedAt: null})
        groupsWithEmailAccess = groupsWithEmailAccess.map(group => group._id)
        
        // Prepare filter for email users
        let emailUserFilter = {
            ...condition,
            groups: { $in: groupsWithEmailAccess }
        };
        let emailUsersCount = await db.User.countDocuments(emailUserFilter)
        let emailUsers = await db.User.find(emailUserFilter)
            .skip((page-1)*limit)
            .limit(limit)
        
        let groupsWithMessengerAccess = await db.Group.find({companyId, messengerAccess: true, deletedAt: null})
        groupsWithMessengerAccess = groupsWithMessengerAccess.map(group => group._id)

        // Prepare filter for whatsapp users
        let whatsappUserFilter = {
            ...condition,
            groups: { $in: groupsWithMessengerAccess }
        };
        let whatsappUsersCount = await db.User.countDocuments(whatsappUserFilter)
        let whatsappUsers = await db.User.find(whatsappUserFilter)
            .skip((page-1)*limit)
            .limit(limit)


        return res.status(200).send({ Success: true, users, count, emailUsers, emailUsersCount, whatsappUsers, whatsappUsersCount })
    } catch (error) {
        console.log('error', error);
        next(error)
    }
}

const apiRouter = express.Router();
apiRouter.route('/').get(jwtStrategy, permission('Users','Read'), validateBody(validator), controller);
export default apiRouter;