import Joi from 'joi';
import express from 'express';

import { checkCompanyValidityStrategy, jwtStrategy } from '../../middlewares/strategy';
import { validateBody } from '../../middlewares/validator';
import { permission } from '../../middlewares/permission';
import { htmlsanitize } from '../../middlewares/sanitizeHtml';
import restrictFeature from '../../middlewares/featureBlocker';

const validator = Joi.object().keys({
    email: Joi.string().required(),
    page: Joi.number().min(0).required(),
    limit: Joi.number().min(1).required(),
    domainId: Joi.string().hex().length(24).optional(),

});

/**
 * @api {post} /api/v1/users/user-list?page=1&limit=10 user-list
 * @apiName user-list
 * @apiGroup users
 * @apiDescription Fetch the list of all-users-tpir
 * 
 * @apiVersion 1.0.0
 * 
 * @apiQuery   {Number}   page          (Pagination) Required page of the events records
 * @apiQuery   {Number}   limit         (Pagination) Required limit of event records to be returned
 * @apiQuery   {String}   domainId      Id of the domain who users are to be fetched
 * @apiQuery   {String}   [search]      Search for a particulat user
 * 
 * @apiSuccess {String} Success             true
 * @apiSuccess {Array}  users               list of all-users-tpir
 * @apiSuccess {Number} count               count of users
 * 
 * @apiSuccessExample Success-Response:
 *      HTTP/1.1 200 OK 
 * {
 *           "Success": true,
 *           "users": [
 *               {
 *                   "_id": "63d0efab032f943fc31e4ebe",
 *                   "name": "admintab",
 *                   "email": "admintab@kdmarc.com",
 *                   "userId": "193848ac-8363-482d-8578-2574b51c7d7d",
 *                   "domainId": "63d0eaaa032f943fc31e4ea2",
 *                   "companyId": "63d0ea61032f943fc31e4e80",
 *                   "isActive": true,
 *                   "emailType": "Outlook",
 *                   "deletedAt": null,
 *                   "__v": 0,
 *                   "createdAt": "2023-01-25T09:00:27.105Z",
 *                   "updatedAt": "2023-01-25T09:00:27.105Z"
 *               }
 *               
 *           ],
 *           "count": 1
 *           }
 * 
 * @apiError {Boolean} Success             false
 * @apiError {Array}   messages            "\"reportedMailId\" is not allowed to be empty"
 * 
 * @apiSuccessExample Error-Response:
 *       HTTP/1.1 400 Not Found
 * 
 *      "success":   false,
 *      "messages" : "\"reportedMailId\" is not allowed to be empty"
 * 
 */
const getReportedMailsWithCount = (reportedMails) => {
    const countsByDate = {};
    reportedMails.forEach(email => {
      const date = new Date(email.createdAt);
      if (!isValidDate(date)) {
        return; // Skip invalid dates
      }
      const formattedDate = date.toISOString().split('T')[0]; // Extracting date in 'YYYY-MM-DD' format
  
      if (countsByDate[formattedDate]) {
        countsByDate[formattedDate]++;
      } else {
        countsByDate[formattedDate] = 1;
      }
    });

    // Create a range of dates between the earliest and latest dates
    const earliestDate = new Date().setMonth(new Date().getMonth() - 3)
    const latestDate = new Date();
    const dateRange = getDateRange(earliestDate, latestDate);
  
    // Fill in missing dates with 0 counts immediately after the date with a record
    const result = {};
    let lastCountedDate = null;
  
    dateRange.forEach(date => {
      const formattedDate = date.toISOString().split('T')[0];
      result[formattedDate] = countsByDate[formattedDate] || 0;
  
    //   if (countsByDate[formattedDate] !== undefined) {
    //     lastCountedDate = formattedDate;
    //   } else if (lastCountedDate !== null) {
    //     result[formattedDate] = 0;
    //   }
    });
    const mails = Object.entries(result).map(([date, count]) => {
      const unixTimestamp = new Date(date).getTime(); // Convert to Unix timestamp
      return [unixTimestamp, count];
    });
  
    const dates = Object.keys(result);
    const latestFormattedDate = dates[dates.length - 1];
    return { mails, latestDate: latestFormattedDate };
  }
  
  // Helper function to check if a date is valid
  function isValidDate(d) {
    return d instanceof Date && !isNaN(d);
  }
  
  // Helper function to get a range of dates between two dates
  function getDateRange(startDate, endDate) {
    const dateRange = [];

    let currentDate = new Date(startDate)
    while (currentDate.toISOString().split('T')[0] <= endDate.toISOString().split('T')[0]) {
      dateRange.push(new Date(currentDate));
      currentDate.setDate(currentDate.getDate() + 1);;
    }
    return dateRange;
  }

  const calculatePercentage = (deletedMails, totalMails) =>{
        if (totalMails !== 0) {
          return (deletedMails / totalMails * 100).toFixed(1);
        } else if (totalMails === 0) {
            return 0;
        } else {
            return 0;
        }
  }
  

const controller = async (req, res, next) => {
    try {
        const { email, page, limit, phoneNumber } = req.body
        let user;

        if(email) user = await db.User.findOne({email, companyId: req.user.companyId}).select('name email emailType isActive isActiveMessenger');
        if(!email && phoneNumber != null) user = await db.User.findOne({phoneNumber, companyId: req.user.companyId}).select('name email emailType isActive isActiveMessenger');

        if(!user) throw new RequestError("User not found!")

        const threeMonthsAgo = new Date();
        threeMonthsAgo.setMonth(threeMonthsAgo.getMonth() - 3);

        const reportedMails = await db.ReportedMail.find({userId:user._id, companyId: req.user.companyId, createdAt:{ $gte: threeMonthsAgo }}).sort({ createdAt: -1 }).select('from subject reporters createdAt isRecoveredEmail isTrashedEmail isDeletedEmail -_id')
        .skip((page-1)*limit)
        .limit(limit)

        const reportedMessages = await db.whatsapp.find({userId:user._id, companyId: req.user.companyId, createdAt:{ $gte: threeMonthsAgo }}).sort({ createdAt: -1 }).select('from phoneNumber sender createdAt status result messageSummary -_id')
        .skip((page-1)*limit)
        .limit(limit)    
        

        const TotalreportedMails = await db.ReportedMail.find({userId:user._id, companyId: req.user.companyId, createdAt:{ $gte: threeMonthsAgo }}).sort({ createdAt: 1 }).select('from subject reporters createdAt isRecoveredEmail isTrashedEmail isDeletedEmail -_id')
        const TotalRportedMessages = await db.whatsapp.find({userId:user._id, companyId: req.user.companyId, createdAt:{ $gte: threeMonthsAgo }}).sort({ createdAt: 1 }).select('from phoneNumber sender createdAt status result messageSummary -_id')
        const count = await db.ReportedMail.countDocuments({userId:user._id, companyId: req.user.companyId, createdAt:{ $gte: threeMonthsAgo }})
        const messengerCount = await db.whatsapp.countDocuments({userId:user._id, companyId: req.user.companyId, createdAt:{ $gte: threeMonthsAgo }}).sort({ createdAt: 1 }).select('from phoneNumber sender createdAt status result messageSummary -_id')
        const isAdmin = await db.Admin.findOne({email, companyId: req.user.companyId})

        const totalMails = await db.ReportedMail.countDocuments({ userId:user._id, companyId: req.user.companyId })
        const deletedMails = await db.ReportedMail.countDocuments({ userId:user._id, companyId: req.user.companyId, isDeletedEmail: true})
        const trashedMails = await db.ReportedMail.countDocuments({ userId:user._id, companyId: req.user.companyId, isTrashedEmail: true})
        const recoveredMails = await db.ReportedMail.countDocuments({ userId:user._id, companyId: req.user.companyId, isRecoveredEmail: true})
        const pendingMails = await db.ReportedMail.countDocuments({ userId:user._id, companyId: req.user.companyId, isDeletedEmail: false, isTrashedEmail: false, isRecoveredEmail: false})

        const totalMessages = await db.whatsapp.countDocuments({ userId:user._id, companyId: req.user.companyId })
        const safeMessages = await db.whatsapp.countDocuments({ userId:user._id, companyId: req.user.companyId, result: 'safe'})
        const unsafeMessages = await db.whatsapp.countDocuments({ userId:user._id, companyId: req.user.companyId, result: 'unsafe'})
        const dubiousMessages = await db.whatsapp.countDocuments({ userId:user._id, companyId: req.user.companyId, result: 'dubious'})
        const noAdminActionMessages = await db.whatsapp.countDocuments({ userId:user._id, companyId: req.user.companyId, result: 'No Action'})

        var userStats = {}
        userStats.name = user.name
        userStats.email = user.email
        userStats.phoneNumber = user.phoneNumber
        userStats.emailType = user.emailType 
        userStats.role = isAdmin ? 'Admin' : 'Member'
        userStats.isActive = user.isActive 
        userStats.totalMails = totalMails
        userStats.deletedMails = deletedMails
        userStats.trashedMails = trashedMails
        userStats.recoveredMails = recoveredMails
        userStats.pendingMails = pendingMails
        userStats.score = deletedMails!= 0 ? calculatePercentage(deletedMails , totalMails) : 0
        userStats.messengerScore = unsafeMessages!=0 ? calculatePercentage(unsafeMessages , totalMessages) : 0

        let reportedMailsWithCount = reportedMails.length ? getReportedMailsWithCount(TotalreportedMails) : {mails:[], latestDate:''}
        let reportedMessagesWithCount = reportedMessages.length ? getReportedMailsWithCount(TotalRportedMessages) : {mails:[], latestDate:''}
        userStats.reportedMailsWithCount = reportedMailsWithCount.mails
        userStats.reportedMessagesWithCount = reportedMessagesWithCount.mails

        userStats.lastActive = reportedMailsWithCount.latestDate == '' ? "No mail reported yet." : reportedMailsWithCount.latestDate
        userStats.lastActiveMessenger = reportedMessagesWithCount.latestDate == '' ? "No message reported yet." : reportedMessagesWithCount.latestDate

        userStats.safeMessages = safeMessages
        userStats.unsafeMessages = unsafeMessages
        userStats.totalMessages = totalMessages
        userStats.dubiousMessages = dubiousMessages
        userStats.noAdminActionMessages = noAdminActionMessages
        userStats.isActiveMessenger = user.isActiveMessenger

        return res.status(200).send({ success: true, userStats, reportedMails, reportedMessages, count, messengerCount })

    } catch (error) {
        console.log('error', error);
        next(error)
    }
}

const apiRouter = express.Router();

apiRouter.route('/').post(htmlsanitize(), jwtStrategy, restrictFeature("TPIR-UAS"), checkCompanyValidityStrategy, permission('Users', 'Read'), 
// validateBody(validator),
 controller);
export default apiRouter;