import express from 'express';
import { sanitize } from '../../middlewares/sanitizer';
import { checkCompanyValidityStrategy, jwtStrategy } from '../../middlewares/strategy';
import restrictFeature from '../../middlewares/featureBlocker';

let controller = async (req, res, next) => {
    try {
        const companyId = req.user.companyId;

        const dailyReportedMessagesPromise = db.whatsapp.aggregate([
            { $match: { companyId } },
            {
                $group: {
                    _id: {
                        $dateToString: { format: "%Y-%m-%d", date: "$createdAt" }
                    },
                    count: { $sum: 1 }
                }
            },
            {
                $project: {
                    createdAt: '$_id',
                    count: 1,
                    _id: 0
                }
            },
            { $sort: { createdAt: 1 } }
        ]).then(reportedMessage => reportedMessage.map(message => {
            const unixTimestamp = new Date(message.createdAt).getTime();
            return [unixTimestamp, message.count];
        }));

        const topUrlsPromise = db.Url.find({
            score: { $gt: 0 },
            reportedMailId: null,
            whatsappMessageId: { $ne: null },
            companyId: companyId
        })
        .select("url score")
        .populate({
            path: "whatsappMessageId",
            select: "from phoneNumber score"
        })
        .sort({ score: -1 })
        .limit(5);

        const whatsappReportedPromise = db.whatsapp.find({
            companyId,
            deletedAt: null,
        })
        .select('from phoneNumber')
        .sort({'createdAt': -1})
        .limit(5);

        const topReportersPromise = db.whatsapp.aggregate([
            { $match: { companyId } },
            {
                $group: {
                    _id: {
                        phoneNumber: "$phoneNumber",
                        name: "$from"
                    },
                    count: { $sum: 1 }
                }
            },
            { $sort: { count: -1 } },
            { $limit: 5 },
            {
                $project: {
                    _id: 0,
                    phoneNumber: "$_id.phoneNumber",
                    name: "$_id.name",
                    count: 1
                }
            }
        ]);

        const topSendersPromise = db.whatsapp.aggregate([
            { $match: { companyId } },
            {
                $group: {
                    _id: {
                        sender: "$sender",
                    },
                    count: { $sum: 1 }
                }
            },
            { $sort: { count: -1 } },
            { $limit: 5 },
            {
                $project: {
                    _id: 0,
                    sender: "$_id.sender",
                    count: 1
                }
            }
        ]);

        const recentUnsafePromise = db.whatsapp.find({companyId, result:"unsafe"})

        const messagesReportedCountPromise = db.whatsapp.countDocuments({companyId})
        const whatsappUsersCountPromise = db.User.countDocuments({companyId, phoneNumber: { $ne: null}})

        const safeMessagesCountPromise = db.whatsapp.countDocuments({companyId, result:"safe"})
        const unsafeMessagesCountPromise = db.whatsapp.countDocuments({companyId, result:"unsafe"})
        const dubiousMessagesCountPromise = db.whatsapp.countDocuments({companyId, result:"dubious"})


        // Execute all promises concurrently
        const [dailyReportedMessages, topUrls, whatsappReported, topReporters, topSenders, messagesReportedCount, whatsappUsersCount, recentUnsafeMessages, safeMessagesCount, unsafeMessagesCount, dubiousMessagesCount ] = await Promise.all([
            dailyReportedMessagesPromise,
            topUrlsPromise,
            whatsappReportedPromise,
            topReportersPromise,
            topSendersPromise,
            messagesReportedCountPromise,
            whatsappUsersCountPromise,
            recentUnsafePromise,
            safeMessagesCountPromise,
            unsafeMessagesCountPromise,
            dubiousMessagesCountPromise
        ]);


        // Combine results into a single response
        res.status(200).json({
            success: true,
            dailyReportedMessages,
            topUrls,
            whatsappReported,
            topReporters,
            topSenders,
            messagesReportedCount,
            whatsappUsersCount,
            recentUnsafeMessages,
            safeMessagesCount,
            unsafeMessagesCount,
            dubiousMessagesCount
        });

    } catch (err) {
        console.error(err);
        next(err);
    }
}

const apiRouter = express.Router();
apiRouter.route('/').get(sanitize(), jwtStrategy, restrictFeature("TPIR-WA"), controller);
export default apiRouter;
