import { get } from 'stack-trace';

global.RequestError =  class RequestError extends Error {
    copyObject(requestError) {
        this.errorList = requestError.errorList;
    }

    constructor(message, code, realError) {
        if(realError instanceof RequestError){
            super(realError.message, realError.code)
            this.copyObject(realError)
            return;
        }
        if(!code)
            code = 500;
		super(message, code);
		this.status = code;
        this.errorList = [];
        if(message instanceof Array){
            for(let i=0;i<message.length;i++){
                this.errorList.push(message[i])
            }
        }else{
            this.errorList.push(message);
        }
        const trace = get();
        var consoleMessage = message;
        console.log(message);
        if(realError)
                consoleMessage = realError;
        console.error('\x1b[31mRequestError\x1b[0m', '\x1b[35m'+trace[1].getFileName().replace(__dirname, '')+'\x1b[0m', '\x1b[32m'+trace[1].getLineNumber()+':'+trace[1].getColumnNumber()+'\x1b[0m', consoleMessage);
    }
};

global.handleError = (arg1) => {
    if (typeof arg1 === 'boolean' && arg1 == true) {
        return async function () { };
    } else if (arg1 instanceof Error) {
        console.error('errorHandler', arg1.message)
    } else {
        console.error('errorHandler', arg1)
    }
}

export class AzureErrorHandler extends Error {
  /**
   * Creates a new Azure error handler
   * @param {string} message - The main error message
   * @param {Object} options - Options for the error
   * @param {string} options.operationType - Type of operation that failed (e.g., 'token', 'email')
   * @param {Object} options.error - The original error object
   * @param {number} options.code - HTTP status code (defaults to 500)
   */
  constructor(message, { operationType, error, code = 500, logInternally=true } = {}) {
    super(message);
    
    this.name = 'AzureErrorHandler';
    this.status = code;
    this.operationType = operationType || 'unknown';
    this.errorDetails = {
      message: message,
      operationType: this.operationType,
      originalMessage: error?.response?.data?.error?.message,
      directDisplayMessage: error?.response?.data?.error?.message || message,
      responseStatus: error?.response?.status,
      responseStatusText: error?.response?.statusText,
      responseData: error?.response?.data ? JSON.stringify(error?.response?.data) : undefined,
      requestUrl: error?.config?.url,
      originalMessage: error?.message,
      stack: error?.stack
    };

    // Log the error with relevant details
    const trace = get();
    logInternally && console.error(
      '\x1b[31mAzureError\x1b[0m',
      '\x1b[34m[' + this.operationType + ']\x1b[0m',
      '\x1b[35m' + (trace[1]?.getFileName()?.replace(__dirname, '') || 'unknown') + '\x1b[0m',
      '\x1b[32m' + (trace[1] ? trace[1].getLineNumber() + ':' + trace[1].getColumnNumber() : '') + '\x1b[0m',
      message,
      error?.message ? `\n\x1b[33mOriginal error:\x1b[0m ${error.message}` : ''
    );
  }

  /**
   * Customizes error serialization for API responses
   * @returns {Object} Serialized error object
   */
  toJSON() {
    return {
      error: true,
      message: this.message,
      operationType: this.operationType,
      status: this.errorDetails.responseStatus || this.status,
      details: this.errorDetails.responseData 
        ? JSON.parse(this.errorDetails.responseData) 
        : { message: this.errorDetails.originalMessage }
    };
  }
}

export class NodemailerError extends Error {
  constructor(message, { error, logInternally}) {
    super(message);
    this.name = 'NodemailerError';
    let directDisplayMessage = message;
    switch(error?.code) {
      case 'EAUTH':
        directDisplayMessage = 'Authentication failed. Please check your SMTP credentials (username/password).';
        break;
      case 'ECONNREFUSED':
        directDisplayMessage = 'Connection refused. This issue could be caused due to misconfigured server host or port.';
        break;
      case 'ENOTFOUND':
        directDisplayMessage = 'SMTP server not found. Please check your SMTP host.';
        break;
      case 'ETIMEDOUT':
        directDisplayMessage = 'Connection timed out. This issue could be caused due to network connectivity problems or misconfigured SMTP server settings.';
        break;
      case 'EENVELOPE':
        directDisplayMessage = 'Invalid email envelope. Please check the recipient email address.';
        break;
      case 'EHOSTUNREACH':
        directDisplayMessage = 'Host unreachable. Please check your SMTP server settings.';
        break;
      case 'ESOCKET':
        directDisplayMessage = 'Socket error occurred. Please check your network connection and SMTP server settings.';
        break;
      case 'ECONNECTION':
        directDisplayMessage = 'Connection error occurred. Please check if a network connection can be established to the given host.';
    } 
    this.errorDetails = {
      message: message,
      originalMessage: error?.message,
      directDisplayMessage,
      stack: error?.stack
    };

    if(logInternally) {
      const trace = get();
      console.error(
        '\x1b[31mNodemailerError\x1b[0m',
        '\x1b[35m' + (trace[1]?.getFileName()?.replace(__dirname, '') || 'unknown') + '\x1b[0m',
        '\x1b[32m' + (trace[1] ? trace[1].getLineNumber() + ':' + trace[1].getColumnNumber() : '') + '\x1b[0m',
        message,
        error?.message ? `\n\x1b[33mOriginal error:\x1b[0m ${error.message}` : ''
      );
    }
  }
}
