import 'dotenv/config';
import path from 'path';
import cors from 'cors';
import {createClient} from 'redis'
import helmet from "helmet";
import morgan from 'morgan';
import express from 'express';
import winston from 'winston';
import passport from 'passport';
import mongoose from 'mongoose';
import bodyParser from 'body-parser';
import session from 'express-session';
import cookieParser from 'cookie-parser';
import rateLimit from 'express-rate-limit';
import expressSanitizer from 'express-sanitizer';

import kue from './kue';
import config from './config';
import cronJob from './cronJob';
import { restRouter } from './api';
import policy from './iam-policy/policy';
import WorkerCon from './workerpool/workerpool';
import isPortReachable from './service/check-port';
import { LogTransport } from './s3logs/LogTransport';
import scheduler from './utils/scheduler';

import './errors';               
import './models';
import './passport';
import { decryptRequest } from './service/decrypt-request';
import featureBlocker from './middlewares/featureBlocker';

const PORT = config.app.port;
global.appRoot = path.resolve(__dirname); 
global.redisClient = createClient({
    host: config.redis.host,
    port: config.redis.port,
    auth: config.redis.password,
  })
try {
	const logger = winston.createLogger({
		transports: [new LogTransport({
			dirname: 'logs',
			filename: 'application-%DATE%.log',
			datePattern: 'YYYY-MM-DD-HH',
			zippedArchive: true,
			maxSize: '20m',
			maxFiles: '14d'
		})]
	});
	global.logger = logger;
} catch (err) { 
    console.log(err); 
    global.logger = { info: () => {}};
}

const app = express();
app.use(cors({ origin: config.app.url, credentials: true }));
app.use(morgan('tiny'));

app.use(express.json({ limit: '50mb' }));
app.use(express.urlencoded({ extended: true, limit: '50mb' }));
app.use(bodyParser.json({ limit: '50mb' }));
app.use(bodyParser.urlencoded({extended: true, limit: '50mb' }));

app.use(cookieParser(config.app.secret));
app.use(session({secret: config.app.secret, resave: true, saveUninitialized: true}));
app.use(passport.initialize());
app.use(expressSanitizer());
app.use(passport.session());           

app.use(
    helmet.contentSecurityPolicy({
        directives: {
            defaultSrc: ["'self'"],
            scriptSrc: ["'self'"],
            imgSrc: ["'self'", "data:", "https:", "https://stag-tpir.threatcop.com/"],
            styleSrc: ["'self'", "https:"],
            fontSrc: ["'self'", "https:", "data:"],
            objectSrc: ["'none'"],
            upgradeInsecureRequests: [],
            frameAncestors: ["'self'"]
        },
    })
);
// app.use(helmet.frameguard({ action: "sameorigin" })); // Blocks all iframes (Best security)

app.use(helmet.referrerPolicy({ policy: "strict-origin-when-cross-origin" }));
app.use(helmet.hidePoweredBy())
app.use(
    helmet.hsts({
        maxAge: 63072000, // Must be in seconds. This is 1 year.
        includeSubDomains: true, // Must be enabled to be approved by preload.
        preload: true, // If you want to include your site in the HSTS preload list.
    })
);

const apiLimiter = rateLimit({
    windowMs: 1 * 60 * 1000, // 1 minutes
    max: 300,
    message:
      "Too many requests created from this IP, please try again after an minute",
  });

/* Route handling */
app.use(apiLimiter)
app.use('/api/v1', decryptRequest);
app.use('/api/v1', restRouter);
app.use('/api-doc', express.static('./apidoc/'))
app.use('/api/static', express.static(path.join(__dirname, 'public')));
app.use('/', (req, res, next) => {
    next(new RequestError('Invalid route',405))
});

app.use((error, req, res, next) => {
    if (!(error instanceof RequestError)) {
        error = new RequestError('something went wrong', 500, error);
    }
    res.status(error.status || 500);
    res.json({ success: false, messages: error.errorList });
});
/* database connection */
mongoose.connect(config.db.connection, {
    useNewUrlParser: true, 
    useUnifiedTopology: true,
    useCreateIndex: true,
    useUnifiedTopology: true
})
.then( async () => {
    kue.init();
    policy.init();
    cronJob.init();
    WorkerCon.init({ 
        maxWorkers: config.worker.limit,
        workerType: config.worker.type 
    });
    await redisClient.connect()
    await scheduler.init();;
    console.log("\x1b[32mDatabase:\x1b[0m", "Database connected successfully.")
    const spamassession = await isPortReachable(config.spamassession.port, { host: config.spamassession.host });
    spamassession? console.log("\x1b[32mSpamassession:\x1b[0m", 'Spamassession connected successfully'): console.error('\x1b[31mSpamassession:\x1b[0m', 'something went wrong with spamassession connection');
    logger.info(`Server started!`);
})    
.catch(err => console.log("Something went wrong Database connection.", err))

/* Start Listening service */
app.listen(PORT, () => {
    console.log("\x1b[32mServer:\x1b[0m", `Server is running at PORT ${PORT}`);
});
