import fs from 'fs';
import path from 'path';
import Transport from 'winston-transport';
import { S3StreamLogger } from 's3-streamlogger';

import config from '../config';

export class LogTransport extends Transport {

	isValidFileName(filename) {
		// eslint-disable-next-line no-control-regex
		return !/["<>|:*?\\/\x00\x01\x02\x03\x04\x05\x06\x07\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f\x10\x11\x12\x13\x14\x15\x16\x17\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f]/g.test(filename);
	}

	isValidDirName(dirname) {
		// eslint-disable-next-line no-control-regex
		return !/["<>|\x00\x01\x02\x03\x04\x05\x06\x07\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f\x10\x11\x12\x13\x14\x15\x16\x17\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f]/g.test(dirname);
	}

	generateStream(clientId) {
		let streamDir = this.dirname;
		if (clientId) streamDir = path.join(streamDir, clientId.toString());
		else streamDir = path.join(streamDir, 'main');

		return new S3StreamLogger({
			name_format: 'app-log-%Y-%m-%d-%H-%M.log',
			rotate_every: 60*60*1000*5, //5 hours
			upload_every: 1000,
			folder: streamDir,
			bucket: config.s3.logBucket,
			access_key_id: config.s3.accessKey,
			secret_access_key: config.s3.secretAccessKey
		});
	}

	getMaxSize(size) {
		if (size && typeof size === 'string') {
			var _s = size.toLowerCase().match(/^((?:0\.)?\d+)([k|m|g])$/);
			if (_s) {
				return size;
			}
		} else if (size && Number.isInteger(size)) {
			var sizeK = Math.round(size / 1024);
			return sizeK === 0 ? '1k' : sizeK + 'k';
		}

		return null;
	}

	constructor(opts) {
		super(opts);
		
		this.availableClientList = {};
		this.options = opts;
		this.filename = opts.filename ? path.basename(opts.filename) : 'winston.log';
		this.dirname = opts.dirname || path.dirname(opts.filename);

		if (!fs.existsSync(this.dirname)) fs.mkdirSync(this.dirname, { recursive: true });
		
		if (!this.isValidFileName(this.filename) || !this.isValidDirName(this.dirname)) {
			throw new Error('Your path or filename contain an invalid character.');
		}

		this.logStream = this.generateStream();
	}

	close(){
		if (this.logStream) {
			this.logStream.end();
		}
	}

	async trackEvent(tracker, event, user, companyIdIn){

		let companyId = companyIdIn;
		if (user) companyId = user.companyId; 

		//Validation
		if(!companyId || !event) throw new Error('Invalid Event to track');

		//Tell Information about User
		if (user) tracker.people.set(user.id, {
			$first_name: user.firstName,
			$last_name: user.lastName,
			$created: user.createdAt,
			$email: user.email,
		});

		//Create Tracking Event
		tracker.track(event, {
			distinct_id: companyId
		});

	}

	async log(info, callback) {
		try {
			if(!info.track) info.track = true;
			let logTime = new Date().toLocaleString();
			let logObj = {
				timestamp: logTime,
				level: info.level,
				details: {
					message: info.message
				}
			};

			//Track Event
			if (info.request) {
				logObj.route = info.request.url;
				logObj.ip = info.request.headers['x-forwarded-for'] || info.request.socket.remoteAddress;

				if (info.track) await this.trackEvent(info.request.tracker, info.event, info.request.user, info.client ? info.client : null).catch(handleError(true));
			}

			//Generate Customer specific log
			if (info.client) {
				if (!this.availableClientList[info.client]) {
					this.availableClientList[info.client] = this.generateStream(info.client);
				}
				this.availableClientList[info.client].write(`${JSON.stringify(logObj)}\n`);
				logObj.companyId = info.client;
			}

			//Write to common logfile
			this.logStream.write(`${JSON.stringify(logObj)}\n`);

			callback();
		} catch (e) {
			console.log(e)
			callback(e);
		}
	}
};