// services/AzureSmtpClient.js
import {promises as fs} from 'fs';
import path from 'path';
import getToken from './get-outlook-token.js';
import triggerMailAzure from './trigger-mail-azure.js';

const constants = {
  redisSafeCacheTimeDeduction: 300, // seconds to subtract before Redis TTL expires
  classSafeCacheTimeDeduction: 5, // seconds to subtract when restoring from Redis
};

function getRedisClientOrThrow() {
  const redisClient = global.redisClient;
  if (!redisClient || !redisClient.isOpen) {
    throw new Error('Redis client is not initialized or not connected.');
  }
  return redisClient;
}

async function fetchBase64FromFile(filename, filePath) {
  const cacheKey = `__base_64_cache_${filename}`;
  let base64String = await getRedisClientOrThrow().get(cacheKey);
  if (!base64String) {
    const buffer = await fs.readFile(filePath);
    base64String = buffer.toString('base64');
    // cache for 24h
    await getRedisClientOrThrow().setEx(cacheKey, 60 * 60 * 24, base64String);
  }
  return base64String;
}

export default class AzureSmtpClient {
  constructor({
    clientId,
    clientSecret,
    tenantId,
    companyId,
    fromAddress,
    isCustom = false,
  }) {
    this.currentApp = isCustom
      ? 'Custom App Registration'
      : 'Native App Registration';
    this.fromAddress = fromAddress;
    this.clientId = clientId;
    this.clientSecret = clientSecret;
    this.tenantId = tenantId;
    this.accessToken = null;
    this.accessTokenExpiry = null;
    this.now = null;
    this.accessTokenRedisKey = [
      'azure_oauth',
      `clientId_${clientId}`,
      `tenantId_${tenantId}`,
      `companyId_${companyId}`,
      `secret_${clientSecret}`,
    ].join('_');
  }

  async populateToken() {
    this.now = Date.now();
    if (
      this.accessToken &&
      this.accessTokenExpiry &&
      this.now < this.accessTokenExpiry
    ) {
      return;
    }
    const found = await this.findTokenInRedis();
    if (!found) {
      await this.initializeAccessToken();
    }
  }

  async initializeAccessToken() {
    const { access_token, expiresIn=360 } = await getToken.getOutlookAuthToken(
      this.tenantId,
      this.clientId,
      this.clientSecret,
    );
    this.accessToken = access_token;
    // subtract a safety margin so we never use a nearly‑expired token
    const ttl = expiresIn - constants.redisSafeCacheTimeDeduction;
    if (ttl > 0) {
      await getRedisClientOrThrow().setEx(this.accessTokenRedisKey, ttl, access_token);
      this.accessTokenExpiry = this.now + ttl * 1000;
    } else {
      // fallback: don't cache
      this.accessTokenExpiry = this.now + expiresIn * 1000;
    }
  }

  async findTokenInRedis() {
    const [token, ttl] = await Promise.all([
      await getRedisClientOrThrow().get(this.accessTokenRedisKey),
      await getRedisClientOrThrow().ttl(this.accessTokenRedisKey),
    ]);
    if (!token || ttl < 0) return false;
    // ttl is in seconds; subtract a small margin before expiration
    this.__helper_setTokenAndExpiry(
      token,
      ttl - constants.classSafeCacheTimeDeduction
    );
    return true;
  }

  __helper_setTokenAndExpiry(token, safeTtlSeconds) {
    this.accessToken = token;
    if (safeTtlSeconds > 0) {
      this.accessTokenExpiry = this.now + safeTtlSeconds * 1000;
    } else {
      this.accessTokenExpiry = null;
    }
  }

  async triggerEmail({
    fromAddress,
    toAddress,
    subject,
    content,
    attachments = [],
    ccAddresses,
    generalHeaders,
    mailerFunction
  }) {
    await this.populateToken();

    // build the Graph payload
    const message = {
      subject,
      body: {
        contentType: 'HTML',
        content,
      },
      toRecipients: [{ emailAddress: { address: toAddress } }],
      ...(ccAddresses && {
        ccRecipients: (Array.isArray(ccAddresses)
          ? ccAddresses
          : [ccAddresses]
        ).map((address) => ({ emailAddress: { address } })),
      }),
      ...(generalHeaders && {
        internetMessageHeaders: Object.entries(generalHeaders).map(
          ([name, value]) => ({ name, value })
        ),
      }),
      ...(attachments.length && {
        attachments:
          await this.__helper_parseAttachmentsForMicrosoftGraphApi(attachments),
      }),
    };
    console.log(`Triggering email for company via ${this.currentApp}`, { mailerFunction });
    const { response: info, timeTaken } = await triggerMailAzure.triggerEmail({
      fromAddress: fromAddress || this.fromAddress,
      payload: { message },
      token: this.accessToken,
    });

    info.customTriggerMessage = `Email sent to ${toAddress} from ${
      fromAddress || this.fromAddress
    } via AzureSmtpClient(${this.currentApp}) in ${timeTaken}s from ${mailerFunction}`;
    info.successResponseTime = timeTaken;
    console.log(info.customTriggerMessage);

    return info;
  }

  async __helper_parseAttachmentsForMicrosoftGraphApi(attachments) {
    const parsed = [];

    for (const a of attachments) {
      let contentBytes;

      if (a.content) {
        contentBytes = Buffer.from(a.content).toString('base64');
      } else if (a.path) {
        contentBytes = await fetchBase64FromFile(a.filename, a.path);
      } else {
        throw new Error(`Attachment must include either 'content' or 'path': ${JSON.stringify(a)}`);
      }

      parsed.push({
        '@odata.type': '#microsoft.graph.fileAttachment',
        name: a.filename,
        contentBytes,
        contentType: a.contentType,
        contentId: a.cid || undefined,
        isInline: a.isInline || false,
      });
    }

    return parsed;
  }
}
