// service/Banner.js
import axios from 'axios';
import Credentials from './credentials';
import updateMessageIdSvc from './update-messageId';
import config from '../config';

// ---------- Banner HTML ----------
export const bannerTemplate = (message) => `
  <div class="threatcop-banner">
    <table cellpadding="0" cellspacing="0" border="0" 
           style="width:100%; max-width:600px; background-color:#fdf1f1; border:2px solid #b83232; border-radius:6px; font-family:Arial, sans-serif;">
      <tr>
        <td style="padding:15px; text-align:left;">
          <!-- Logo -->
          <img src="${config.app.url}/favicon.png"
               alt="Threatcop Logo" 
               width="32" height="32" 
               style="display:block; margin-bottom:10px;" />
          <!-- Text -->
          <p style="margin:0; color:#000; font-size:15px; line-height:1.5;">
            ${message}
          </p>
        </td>
      </tr>
    </table>
    <br/>
  </div>
`;


// extract the exact banner block (div + trailing <br/>) for precise removal
function extractBanner(html) {
  if (typeof html !== 'string') return null;
  const re = /<div[^>]*class=["']threatcop-banner["'][\s\S]*?<\/div>/i;
  const m = html.match(re);
  return m ? m[0] : null;
}
function hasBanner(html) {
  return !!extractBanner(html);
}

// ---------- Graph helpers ----------
async function getAccessToken(companyId, domainId) {
  const { token } = await Credentials.getCredentials(companyId, domainId);
  return token?.access_token || null;
}

export async function getEmailBody(userPrincipalName, messageId, accessToken) {
  const { data } = await axios.get(
    `https://graph.microsoft.com/v1.0/users/${encodeURIComponent(userPrincipalName)}/messages/${messageId}?$select=body`,
    { 
      headers: { Authorization: `Bearer ${accessToken}` },
      ...(config.proxy.enableForMicrosoftGraph && { httpsAgent: config.proxy.proxyAgent, httpAgent: config.proxy.proxyAgent })
    }
  );
  return data?.body?.content || '';
}

export async function updateEmailBody(userPrincipalName, messageId, accessToken, htmlContent) {
  await axios.patch(
    `https://graph.microsoft.com/v1.0/users/${encodeURIComponent(userPrincipalName)}/messages/${messageId}`,
    { body: { contentType: 'HTML', content: htmlContent } },
    { 
      headers: { Authorization: `Bearer ${accessToken}`, 'Content-Type': 'application/json' },
      ...(config.proxy.enableForMicrosoftGraph && { httpsAgent: config.proxy.proxyAgent, httpAgent: config.proxy.proxyAgent })
    }
  );
}

// ---------- Refresh real Graph messageId using your existing service ----------
export async function refreshMessageIdForWhoElse({ whoElseIds, companyId, domainId }) {
  try {
    const { token } = await Credentials.getCredentials(companyId, domainId);
    // call the same way recover API does: an array of WhoElse IDs + token
    await updateMessageIdSvc.processUpdateMessageId(whoElseIds, token);

    // read back updated docs from DB (service bulkWrites messageId)
    const docs = await db.WhoElse
      .find({ _id: { $in: whoElseIds } })
      .select('to messageId isBannerInjected');

    // keep return type as an array (callers can destructure if they passed length=1)
    return docs;
  } catch {
    // on error, still return current state
    return await db.WhoElse
      .find({ _id: { $in: whoElseIds } })
      .select('to messageId isBannerInjected');
  }
}

// ---------- Direct banner ops (when you already have messageId) ----------
export async function injectBanner({ userPrincipalName, messageId, companyId, domainId }) {
  const meta = await db.MetaData.findOne({ companyId, name: 'BannerMessage', deletedAt: null   });
  if (!meta?.value) return;

  const accessToken = await getAccessToken(companyId, domainId);
  if (!accessToken) return;

  const originalBody = await getEmailBody(userPrincipalName, messageId, accessToken);
  if (hasBanner(originalBody)) return; // already there

  const bannerHtml = bannerTemplate(meta.value.trim());
  await updateEmailBody(userPrincipalName, messageId, accessToken, bannerHtml + (originalBody || ''));
}

export async function removeBanner({ userPrincipalName, messageId, companyId, domainId }) {
  const meta = await db.MetaData.findOne({ companyId, name: 'BannerMessage' });
  if (!meta?.value) return;

  const accessToken = await getAccessToken(companyId, domainId);
  if (!accessToken) return;

  const originalBody = await getEmailBody(userPrincipalName, messageId, accessToken);
  const bannerBlock = extractBanner(originalBody);
  if (!bannerBlock) return; // nothing to remove

  const cleaned = originalBody.replace(bannerBlock, '');
  await updateEmailBody(userPrincipalName, messageId, accessToken, cleaned);
}

// ---------- WhoElse-aware ops (refresh messageId via your service first) ----------
export async function injectBannerForWhoElse({ whoElseId, companyId, domainId }) {
  const meta = await db.MetaData.findOne({ companyId, name: 'BannerMessage', deletedAt: null });
  if (!meta?.value) return { changed: false, reason: 'no-banner-meta' };

  // pass array of IDs like recover API
  const docs = await refreshMessageIdForWhoElse({ whoElseIds: [whoElseId], companyId, domainId });
  const refreshed = docs?.[0];
  if (!refreshed?.messageId || !refreshed?.to) {
    return { changed: false, reason: 'no-messageId' };
  }

  const accessToken = await getAccessToken(companyId, domainId);
  if (!accessToken) return { changed: false, reason: 'no-access-token' };

  const body = await getEmailBody(refreshed.to, refreshed.messageId, accessToken);
  if (hasBanner(body)) {
    if (!refreshed.isBannerInjected) {
      await db.WhoElse.findByIdAndUpdate(whoElseId, { isBannerInjected: true });
    }
    return { changed: false, reason: 'already-injected' };
  }

  const bannerHtml = bannerTemplate(meta.value.trim());
  await updateEmailBody(refreshed.to, refreshed.messageId, accessToken, bannerHtml + (body || ''));
  await db.WhoElse.findByIdAndUpdate(whoElseId, { isBannerInjected: true });
  return { changed: true };
}

export async function removeBannerForWhoElse({ whoElseId, companyId, domainId }) {
  const meta = await db.MetaData.findOne({ companyId, name: 'BannerMessage' });
  if (!meta?.value) return { changed: false, reason: 'no-banner-meta' };

  // pass array of IDs like recover API
  const docs = await refreshMessageIdForWhoElse({ whoElseIds: [whoElseId], companyId, domainId });
  const refreshed = docs?.[0];
  if (!refreshed?.messageId || !refreshed?.to) {
    return { changed: false, reason: 'no-messageId' };
  }

  const accessToken = await getAccessToken(companyId, domainId);
  if (!accessToken) return { changed: false, reason: 'no-access-token' };

  const body = await getEmailBody(refreshed.to, refreshed.messageId, accessToken);
  const bannerBlock = extractBanner(body);

  if (!bannerBlock) {
    if (refreshed.isBannerInjected) {
      await db.WhoElse.findByIdAndUpdate(whoElseId, { isBannerInjected: false });
    }
    return { changed: false, reason: 'no-banner-present' };
  }

  const cleaned = body.replace(bannerBlock, '');
  await updateEmailBody(refreshed.to, refreshed.messageId, accessToken, cleaned);
  await db.WhoElse.findByIdAndUpdate(whoElseId, { isBannerInjected: false });
  return { changed: true };
}
