// services/MailService.js

import config from '../config';
import AzureSmtpClient  from './AzureSmtpClient';
import NodemailerSmtpClient from './NodemailerSmtpClient';
import crypto from '../crypto';

export default class MailService {
  constructor(companyId) {
    this.companyId = companyId;
    this.smtpStatus = null; // 'default' | 'custom' | 'microsoft-default' | 'microsoft-custom'
    this.senderType = null; // 'nodemailer' | 'microsoft'
    this.senderConfig = {};
    this.client = null;
    this._azureCred = null;
  }

  /**
   * Bootstraps everything:
   *   - mode detection (azure vs custom vs default)
   *   - loads credentials into this.senderConfig
   */
  async init({ triggerDefault = false } = {}) {
    if (!triggerDefault && this.companyId) {
      await this._detectMode();
    } else {
      this.smtpStatus = 'default';
    }
    await this._loadConfig();
    return this;
  }

  /**
   * Send an email. Validates inputs and forwards to the proper client.
   */
  async triggerEmail({
    recipientAddress,
    subject,
    content,
    ccAddresses,
    generalHeaders,
    attachments,
    mailerFunction
  }) {
    if (!this.client) this._createClient();

    // basic validation
    if (!recipientAddress?.includes('@'))
      throw new RequestError('INVALID_RECIPIENT', 400);
    if (!subject)
      throw new RequestError('MISSING_SUBJECT', 400);

    return this.client.triggerEmail({
      fromAddress: this.senderConfig.fromAddress,
      toAddress: recipientAddress,
      subject,
      content,
      ccAddresses,
      generalHeaders,
      attachments,
      mailerFunction
    });
  }

  // ─── INTERNAL ────────────────────────────────────────────────────────────

  /** Decide between Azure vs custom vs default */
  async _detectMode() {
    // 1) Azure?
    const cred = await db.SmtpCred.findOne({ companyId: this.companyId, deletedAt: null });
    if (cred) {
      this._azureCred = cred;
      this.smtpStatus =
        cred.type === 'Oauth' ? 'microsoft-default' : 'microsoft-custom';
      return;
    }

    // 2) Nodemailer custom?
    const flag = await db.MetaData.findOne({
      name: 'mailSmtp',
      companyId: this.companyId,
    });
    if (flag?.value === 'true') {
      this.smtpStatus = 'custom';
      return;
    }

    // 3) fallback
    this.smtpStatus = 'default';
  }

  /** Based on smtpStatus, load senderType+senderConfig */
  async _loadConfig() {
    switch (this.smtpStatus) {
      case 'default':
        return this._loadDefault();

      case 'custom':
        return this._loadCustom();

      case 'microsoft-default': // oauth
        return this._loadAzure(false);

      case 'microsoft-custom': //manual creds
        return this._loadAzure(true);

      default:
        throw new RequestError(`UNHANDLED_MODE:${this.smtpStatus}`, 500);
    }
  }

  async _loadDefault() {
    this.senderType = 'nodemailer';
    this.senderConfig = {
      host: config.mail.host,
      port: +config.mail.port,
      username: config.mail.username,
      password: config.mail.password,
      fromAddress: config.mail.from,
    };
  }

  async _loadCustom() {
    this.senderType = 'nodemailer';

    // pull exactly the four fields; error if missing
    const keys = ['smtpHost', 'smtpPort', 'smtpUsername', 'smtpPassword'];
    const docs = await Promise.all(
      keys.map((name) =>
        db.MetaData.findOne({ name, companyId: this.companyId, deletedAt: null })
      )
    );
    if (docs.some((d) => !d?.value)) {
      throw new RequestError('SMTP_CONFIG_INCOMPLETE', 400);
    }

    const m = Object.fromEntries(docs.map((d) => [d.name, d.value]));
    const fromDoc = await db.MetaData.findOne({
      name: 'smtpFrom',
      companyId: this.companyId,
      deletedAt: null,
    });

    this.senderConfig = {
      host: m.smtpHost,
      port: +m.smtpPort,
      username: m.smtpUsername,
      password: m.smtpPassword,
      fromAddress: fromDoc?.value || config.mail.from,
    };
  }

  async _loadAzure(isCustom) {
    this.senderType = 'microsoft';
    const c = this._azureCred;

    this.senderConfig = {
      clientId: isCustom
        ? crypto.decrypt(c.clientId)
        : config.azure.defaultMailAppClientId,

      clientSecret: isCustom
        ? crypto.decrypt(c.secretId)
        : config.azure.defaultMailAppClientSecret,

      tenantId: crypto.decrypt(c.tenantId),
      fromAddress: c.senderEmail,
      companyId: this.companyId
    };
  }

  /** Instantiate the actual SMTP client */
  _createClient() {
    if (this.senderType === 'microsoft') {
      this.client = new AzureSmtpClient(this.senderConfig);
    } else {
      this.client = new NodemailerSmtpClient(this.senderConfig);
    }
  }

  getSummary() {
    return {
      senderType: this.senderType,
      smtpStatus: this.smtpStatus,
      fromAddress: this.senderConfig?.fromAddress,
      host: this.senderConfig?.host,
      clientId: this.senderConfig?.clientId,
      tenantId: this.senderConfig?.tenantId,
      companyId: this.companyId,
    }
  }
}