import nodemailer from 'nodemailer';
import { NodemailerError } from '../errors';

export default class NodemailerSmtpClient {
  constructor({
    host,
    port,
    username,
    password,
    ssl = null,
    requiretls = false,
    maxMessages,
    maxConnections,
    connectionTimeout,
    pool,
    fromAddress,
    companyId,
    isCustom,
  }) {
    this.companyId = companyId;
    this.fromAddress = fromAddress;
    this.currentApp = isCustom ? 'Custom SMTP' : 'Native SMTP';
    const transportConfig = {
      host,
      port,
      tls: { rejectUnauthorized: false },
      pool: pool || undefined,
      maxMessages: maxMessages || undefined,
      maxConnections: maxConnections || undefined,
      connectionTimeout: connectionTimeout || undefined,
      requireTLS: requiretls || undefined,
      ...(!requiretls &&
        typeof ssl === 'boolean' ? { secure: ssl, debug: ssl } : {}),
      auth: {
        user: username || undefined,
        pass: password || undefined,
      },
    };
    this.transportConfig = transportConfig;
    this.smtpTransport = nodemailer.createTransport(transportConfig);
    this.invalidSMTP = undefined;
  }

  async triggerEmail({
    fromAddress,
    toAddress,
    subject,
    content,
    attachments = [],
    ccAddresses,
    generalHeaders,
    mailerFunction
  }) {
    try {
      console.log(`Triggering email for company ${this.companyId} via ${this.currentApp}`, { mailerFunction });
      if (this.invalidSMTP === undefined) await this.verifyConnection();
      if (this.invalidSMTP)
        throw new Error('SMTP configuration invalid or connection failed');

      const mailOptions = {
        from: fromAddress || this.fromAddress,
        to: toAddress,
        subject,
        html: content,
        ...(ccAddresses && { cc: ccAddresses }),
        ...(generalHeaders && { headers: generalHeaders }),
        ...(attachments.length > 0 && { attachments }),
      };

      const start = performance.now();
      const info = await this.smtpTransport.sendMail(mailOptions);
      const duration = ((performance.now() - start) / 1000).toFixed(3);

      if (info.rejected?.length) {
        throw new Error(`Rejected recipients: ${info.rejected.join(', ')}`);
      }

      info.successResponseTime = duration;
      info.customTriggerMessage = `Email sent to ${toAddress} from ${mailOptions.from} via NodemailerSmtpClient(${this.currentApp}) in ${duration}s from ${mailerFunction}`;
      console.log(info.customTriggerMessage);

      return info;
    } catch (err) {
      console.error(
        `Error sending email for company ${this.companyId} via NodemailerSmtpClient:`,
        err
      );
      throw new NodemailerError(
        `Failed to send email via Nodemailer for company ${this.companyId}`,
        { error: err, logInternally: true }
      );
    }
  }

  async verifyConnection() {
    try {
      await this.smtpTransport.verify();
      this.invalidSMTP = false;
    } catch (err) {
      this.invalidSMTP = true;
      console.error(
        `SMTP verify failed for company ${this.companyId}:`,
        err,
        this.transportConfig
      );
      throw err;
    }
  }
}
