import mongoose from 'mongoose';
const getTopReportersData = async (condition) => {
        const count = await db.ReportedMail.countDocuments(condition);
        const toCounts = await db.ReportedMail.aggregate([
            { $match:  condition },
            {
                $group: {
                    _id: "$to",
                    count: { $sum: 1 },
                },
            },
        ]);
        
        const reporterCounts = await db.ReportedMail.aggregate([
            { $match: condition  },
            { $unwind: "$reporters" },
            {
                $group: {
                    _id: "$reporters.email", 
                    count: { $sum: 1 },
                },
            },
        ]);
        
        const emailCounts = {};

        const addCounts = (array) => {
            array.forEach(({ _id, count }) => {
                const normalizedEmail = _id.toLowerCase();
                emailCounts[normalizedEmail] = (emailCounts[normalizedEmail] || 0) + count;
            });
        };
    
        addCounts(toCounts);
        addCounts(reporterCounts);
    
        const mergedArray = Object.entries(emailCounts).map(([email, count]) => ({ email, count })).sort((a, b) => b.count - a.count).slice(0, 5)
        const emails = mergedArray.map(({ email }) => email);
        const users = await db.User.find({ email: { $in: emails }, companyId: condition.companyId }).select('name email').lean();
        const userMap = users.reduce((acc, user) => {
            acc[user.email] = user.name || 'Unknown';
            return acc;
        }, {});

        const result = mergedArray.map(({ email, count }) => ({
            name: userMap[email] || 'Unknown', 
            email: email,
            count: count,
        }));

        return {data: result, count, label:'Email Reporting', type:"TableListingType1"};
};

const getHighestReportedEmails = async (condition) => {

    const highestReportedEmails = await db.ReportedMail.aggregate([
        { $match :  condition  },
        {
          $addFields: {
            totalReporters: { $size: "$reporters" } 
          }
        },
        {
          $sort: { totalReporters: -1 } 
        },
        {
          $limit: 5 
        },
        {
          $project: {
            _id: 1,
            totalReporters: 1,
            subject: 1,
            isDeletedEmail: 1,
            isTrashedEmail: 1,
            isRecoveredEmail: 1,
            createdAt: 1,
            from: 1,
            fromName: 1
          }
        }
    ]);

    return {data: highestReportedEmails, label: 'Email Reporting', type:"TPIRTableListingType1"}
};

const getTopSenders = async (condition) => {

  condition = {...condition, isDeletedEmail: "true" }

  const senders = await db.ReportedMail.aggregate([
    {
      $match: condition
    },
    {
      $lookup: {
        from: "whoelses", 
        localField: "_id",
        foreignField: "reportedMailId",
        as: "whoElseDocuments"
      }
    },
    {
      $group: {
        name:{"$first":"$fromName"},
        _id: "$from",
        reportedMailCount: { $sum: 1 },
        whoElseCount: { $sum: { $size: "$whoElseDocuments" } }
      }
    },
    {
        $sort: {
          reportedMailCount: -1 // Sort in descending order
        }
    },
    {
      $limit: 5
    }
  ]);

  return {data: senders, label:"Email Reporting", type: "TPIRTableListingType4" }
};

const getTopVulnerableAttachments = async (condition) => {
  let topAttachments = await db.Attachment.find({...condition, score: { $gte: 1 }})
    .select("originalName score fileName")
    .populate({
      path: "reportedMailId",
      select: "from fromName companyId",
    })
    .sort({ score: -1 }) 
    .limit(5);

    return {data: topAttachments, label:"Email Reporting", type:"TPIRTableListingType3"}
};

const getTopVulnerableLinks = async (condition) => {
  let topLinks = await db.Url.find({...condition, score: { $gte: 1 }})
  .populate({
    path: "reportedMailId",
    select: "from fromName companyId",
  })
  .select("url score companyId")
  .sort({ score: -1 }) 
  .limit(5);
  console.log({topLinks})

    return {data: topLinks, label:"Email Reporting", type:"TPIRTableListingType2"}
};

const getDailyReportedMails = async (condition) => {
      const reportedEmails = await db.ReportedMail.aggregate([
        { $match : condition },
        {
            $group: {
              _id: {
                $dateToString: { format: "%Y-%m-%d", date: "$createdAt" }
              },
              count: { $sum: 1 }
            }
          },
          {
            $project: {
              createdAt: '$_id',
              count: 1,
              _id: 0
            }
          },
          {
            $sort: {
              createdAt: 1
            }
          }
    ]);
    
    const result = reportedEmails.map(mail => {
      const unixTimestamp = new Date(mail.createdAt).getTime();
      return [unixTimestamp, mail.count];
    });

    return {data: result, label:'Email Reporting', type:"AreaChart"}
};

const graphHandlers = {
  RETPIR1: getTopReportersData,
  RETPIR2: getHighestReportedEmails,
  RETPIR3: getTopVulnerableLinks,
  RETPIR4: getTopVulnerableAttachments,
  RETPIR5: getTopSenders,
  RETPIR6: getDailyReportedMails
}
const getGraphData = async (graphCode, params) => {
  const handler = graphHandlers[graphCode];
  if (!handler) {
    throw new Error(`No handler found for graphCode: ${graphCode}`);
  }
  return handler(params);
};

export default getGraphData