import axios from 'axios';
import config from '../config';

/**
 * Fetch attachments from an email.
 */
export const getAttachments = async (email, messageId, accessToken) => {
    try {
        const response = await axios.get(`https://graph.microsoft.com/v1.0/users/${email}/messages/${messageId}/attachments`, {
            headers: {
                Authorization: `Bearer ${accessToken}`,
                Accept: "application/json"
            },
            ...config.proxy.enableForMicrosoftGraph && {
                httpsAgent: config.proxy.proxyAgent,
                httpAgent: config.proxy.proxyAgent
            }
        });

        if (!response.data || !response.data.value.length) {
            console.warn(`No attachments found for message: ${messageId}`);
            return [];
        }

        return response.data.value.map(attachment => ({
            Id: attachment.id,
            Name: attachment.name || "unknown",
            ContentType: attachment.contentType || "application/octet-stream",
            ContentBytes: attachment.contentBytes ? Buffer.from(attachment.contentBytes, 'base64') : null
        }));
    } catch (error) {
        console.error("Error fetching attachments:", error.response?.data || error.message);
        return [];
    }
};



export const getOriginalMessageDetails = async (email, messageId, attachmentId, accessToken) => {
    try {
        const response = await axios.get(`https://graph.microsoft.com/v1.0/users/${email}/messages/${messageId}/attachments/${attachmentId}/$value`, {
            headers: {
                Authorization: `Bearer ${accessToken}`,
                Accept: "message/rfc822"
            },
            responseType: 'arraybuffer',
            ...config.proxy.enableForMicrosoftGraph && {
                httpsAgent: config.proxy.proxyAgent,
                httpAgent: config.proxy.proxyAgent
            }
        });

        return response.data.toString('utf8');
    } catch (error) {
        console.error("Error fetching original message details:", error.response?.data || error.message);
        return null;
    }
};




export const getSenderEmail = async (email, messageId, accessToken) => {
    try {
        const getMessageUrl = `https://graph.microsoft.com/v1.0/users/${email}/messages/${messageId}`;

        const response = await axios.get(getMessageUrl, {
            headers: {
                Authorization: `Bearer ${accessToken}`,
                Accept: "application/json"
            },
            ...config.proxy.enableForMicrosoftGraph && {
                httpsAgent: config.proxy.proxyAgent,
                httpAgent: config.proxy.proxyAgent
            }
        });

        if (!response.data || !response.data.from || !response.data.from.emailAddress) {
            console.error(" No valid sender information found in the email.");
            return null;
        }

        const senderEmail = response.data.from.emailAddress.address;
        console.log("Extracted 'From' Email:", senderEmail);

        return senderEmail;
    } catch (error) {
        console.error("Error fetching email:", error.response?.data || error.message);
        return null;
    }
};
