import path from 'path';
import axios from 'axios';
import credentials from '../service/credentials';
import config from '../config';

const GRAPH_API_URL = "https://graph.microsoft.com/v1.0/subscriptions";
const MAX_SUBSCRIPTION_MINUTES = 10070; 


const getExistingSubscription = async (email, token) => {
    try {
        const response = await axios.get(GRAPH_API_URL, {
            headers: { Authorization: `Bearer ${token}` },
            ...(config.proxy.enableForMicrosoftGraph && { httpsAgent: config.proxy.proxyAgent, httpAgent: config.proxy.proxyAgent })
        });

        if (response.data && response.data.value) {
            return response.data.value.find(sub => sub.resource.includes(email));
        }
        return null;
    } catch (error) {
        console.error("Error fetching existing subscriptions:", error.message);
        return null;
    }
};


const deleteSubscription = async (subscriptionId, token) => {
    try {
        await axios.delete(`${GRAPH_API_URL}/${subscriptionId}`, {
            headers: { Authorization: `Bearer ${token}` },
            ...(config.proxy.enableForMicrosoftGraph && { httpsAgent: config.proxy.proxyAgent, httpAgent: config.proxy.proxyAgent })
        });
        console.log(`Deleted existing subscription: ${subscriptionId}`);
    } catch (error) {
        console.error(`Error deleting subscription ${subscriptionId}:`, error.message);
    }
};


export const createSubscription = async (companyId, domainId, email) => {
    try {
        const credential = await credentials.getCredentials(companyId, domainId);
        if (!credential ) {
            throw new Error(`Invalid credentials!`);
        }

        let token = credential?.token?.access_token;
        if(!token){
            throw new Error(`Invalid credentials!`);
        }
        const existingSub = await getExistingSubscription(email, token);

        if (existingSub) {
            console.log(`🔹 Found existing subscription (${existingSub.id}), deleting it...`);
            await deleteSubscription(existingSub.id, token); 
        }

        const response = await axios.post(
            GRAPH_API_URL,
            {
                changeType: "created",
                notificationUrl: `${config.app.url}/api/v1/report/webhook-report`,
                resource: `/users/${email}/mailFolders('Inbox')/messages`,
                expirationDateTime: new Date(Date.now() + MAX_SUBSCRIPTION_MINUTES * 60 * 1000).toISOString(),
                clientState: `${companyId}-${domainId}-${email}`
            },
            {
                headers: { Authorization: `Bearer ${token}`, "Content-Type": "application/json" },
                ...(config.proxy.enableForMicrosoftGraph && { httpsAgent: config.proxy.proxyAgent, httpAgent: config.proxy.proxyAgent })
            }
        );

        if (![200, 201].includes(response.status)) {
            throw new Error(`Failed to create subscription: ${response.statusText}`);
        }

        console.log(`New subscription created: ${response.data.id}`);
        return { success: true, message: "Subscription created successfully!", subscriptionId: response.data.id };
    } catch (error) {
        console.error("Subscription creation error:", error.message);
        return { success: false, error: error.message };
    }
};

export const getMessageByInternetMessageId = async (email, internetMessageId, accessToken) => {
    try {
        // Graph API endpoint to search for the email using internetMessageId
        const graphUrl = `https://graph.microsoft.com/v1.0/users/${email}/messages?$filter=internetMessageId eq '${internetMessageId}'`;

        const response = await axios.get(graphUrl, {
            headers: {
                Authorization: `Bearer ${accessToken}`,
                Accept: "application/json"
            },
            ...(config.proxy.enableForMicrosoftGraph && { httpsAgent: config.proxy.proxyAgent, httpAgent: config.proxy.proxyAgent })
        });

        // Check if any messages are returned
        if (response.data.value.length > 0) {
            console.log("Email found:", response.data.value[0]);
            return response.data.value[0]; // Return the first match
        } else {
            console.log("No email found with the given InternetMessageId.");
            return null;
        }
    } catch (error) {
        console.error("Error fetching email:", error.response?.data || error.message);
        throw error;
    }
};

export const formatMailAttachments = (mailAttachments) => {
    console.log({mailAttachments})
    return mailAttachments.map(attachment => ({
        id: attachment.contentId || '', // Use contentId if available
        name: attachment.fileName || 'unknown',
        contentType: attachment.contentType || 'application/octet-stream',
        ContentBytes: attachment.content || null // Buffer content
    }));
};



