import schedule from 'node-schedule';
import processUser from '../service/process-user';
import credentials from '../service/credentials';

const dayMapping = {
  All: 0,
  Monday: 1, Tuesday: 2, Wednesday: 3,
  Thursday: 4, Friday: 5, Saturday: 6, Sunday: 0
};

const scheduler = {
  init: async () => {
    const domains = await db.Domain.find({ 'isScheduledSyncing.isActive': true, deletedAt: null });
    for (const d of domains) {
      const s = d.isScheduledSyncing;
      await scheduler.scheduleDomainAutoSyncSetting({
        domainId:            d._id.toString(),
        companyId:           d.companyId.toString(),
        days:                s.daysOfWeek,
        hour:                s.hour,
        minute:              s.minute,
        timeZone:            s.timeZone,
        syncType:            s.syncType,
        groupIds:            s.groupIds,
        isUserSyncMirrored:  s.isUserSyncMirrored
      });
    }
    console.log(`✔ Scheduler loaded ${domains.length} job(s)`);
  },

  // Cancel any existing job by ID
  cancelScheduledJob: (jobId) => {
    return schedule.cancelJob(jobId.toString());
  },

  // Exactly your inline-scheduler logic, lifted into a reusable fn
  scheduleDomainAutoSyncSetting: async ({
    domainId,
    companyId,
    days = [],
    hour,
    minute,
    timeZone,
    syncType,
    groupIds = [],
    isUserSyncMirrored
  }) => {
    // 1) Tear down any previous job
    scheduler.cancelScheduledJob(domainId);

    // 2) Build the RecurrenceRule
    if (days.includes('All')) {
      days = days.filter(d => d !== 'All');
    }
    const rule = new schedule.RecurrenceRule();
    rule.hour   = Number(hour);
    rule.minute = Number(minute);
    rule.tz     = timeZone;
    if (days.length) {
      rule.dayOfWeek = days.map(d => dayMapping[d]);
    }

    // 3) Schedule it
    schedule.scheduleJob(domainId.toString(), rule, async () => {
      const syncTs = new Date();
      try {
        // ─── A) Fetch & validate domain ──────────────────────────
        const domain = await db.Domain.findOne({ _id: domainId, companyId, deletedAt: null });
        if (!domain) throw new RequestError('Domain does not exist or wrong domain id passed!', 404);
        if (!domain.isVerified) throw new RequestError('Domain is not verified', 400);
        if (domain.isVerified && !domain.isCredential) throw new RequestError('Credentials are not uploaded', 400);
        if (domain.isVerified && domain.isCredential && domain.isSyncingUser)
          throw new RequestError(`Syncing Already in Progress for ${domain.domainName}. Kindly wait until syncing is completed!`, 400);
        if (!domain.isScheduledSyncing?.isActive)
          throw new RequestError('Scheduled Sync is not active for this domain.', 400);

        // ─── B) Prep groups & creds ──────────────────────────────
        const autoSyncGroups = domain.isScheduledSyncing.groupIds || [];
        console.log("domain.isScheduledSyncing------------------------", domain.isScheduledSyncing);

        let subDomainGroups = await db.SubDomainGroup.find(
          { domainId: domain._id, companyId },
          { _id: 1, groupName: 1 }
        );
        subDomainGroups = subDomainGroups.map(g => g.groupName);

        const creds = await credentials.getCredentials(companyId, domainId);

        // ─── C) Find default all-users group ───────────────────
        const allGroup = await db.Group.findOne({
          companyId,
          isDefaultGroup: true,
          deletedAt: null
        });
        if (!allGroup) throw new Error('Default all-users group not found');

        // ─── D) Branch by syncType ─────────────────────────────
        if (autoSyncGroups.length > 0 && syncType === 'specificGroups') {
          console.log({ autoSyncGroups });
          const results = await Promise.allSettled(
            autoSyncGroups.map(async adGroupId => {
              // per-AD group
              const isExchange = domain.domainTypeForOutlook === 'EXCHANGE';
              const groupName  = isExchange ? `${adGroupId.name}-Exchange` : adGroupId.name;
              console.log('groupName:-----------addd', groupName);

              let grp = await db.Group.findOne({
                groupName,
                groupId:   adGroupId._id,
                domainId,
                companyId,
                deletedAt: null
              });
              if (!grp) {
                grp = await db.Group.create({
                  companyId,
                  groupName,
                  type: 'Admin',
                  groupId: adGroupId._id,
                  domainId,
                  isADSynced: true,
                  isSyncingUsers: false
                });
              }

              console.log('isMirror:', domain.isScheduledSyncing.isUserSyncMirrored);
              await processUser.process(
                companyId,
                domainId,
                domain.domainName,
                subDomainGroups,
                domain.domainType,
                false,
                domain.domainTypeForOutlook === 'EXCHANGE',
                grp,
                creds,
                isUserSyncMirrored,
                null,
                syncTs
              );
              return { groupId: grp._id, status: 'ok' };
            })
          );
          console.log('Sync results:', results);

        } else {
          // All‐users branch
          console.log('isMirror:', domain.isScheduledSyncing.isUserSyncMirrored);
          await processUser.process(
            companyId,
            domainId,
            domain.domainName,
            subDomainGroups,
            domain.domainType,
            false,
            domain.domainTypeForOutlook === 'EXCHANGE',
            allGroup,
            creds,
            isUserSyncMirrored,
            allGroup._id,
            syncTs
          );
          await db.Domain.updateOne({ _id: domainId }, { lastAllUsersSyncAt: syncTs });
        }

        // ─── E) Mirror‐cleanup for specificGroups ────────────────
        if (
          autoSyncGroups.length > 0 &&
          syncType === 'specificGroups' &&
          isUserSyncMirrored &&
          !domain.lastAllUsersSyncAt
        ) {
          await db.User.updateMany(
            {
              companyId,
              domainId,
              isADSynced: true,
              groups: allGroup._id,
              lastAdSeenAt: { $ne: syncTs }
            },
            { $pull: { groups: allGroup._id } }
          );
          await db.User.updateMany(
            {
              companyId,
              isADSynced: true,
              deletedAt: null,
              groups: { $size: 0 }
            },
            { $set: { deletedAt: new Date() } }
          );
        }

      } catch (err) {
        console.error('Scheduled sync error:', err);
      }
    });

    console.log(`🔔 Scheduled domain ${domainId} @ ${hour}:${minute} (${timeZone})`);
  }
};

export default scheduler;
