const axios = require('axios');
const { removeStopwords } = require('stopword');
const config = require('../config').default;

const isNonEmpty = (v) => typeof v === 'string' && v.trim().length > 0;

async function fetchAllPages(url, headers) {
  const all = [];
  let next = url;
  const axiosConfig = {
    headers,
    ...(config.proxy.enableForMicrosoftGraph && { httpsAgent: config.proxy.proxyAgent, httpAgent: config.proxy.proxyAgent })
  };
  while (next) {
    const res = await axios.get(next, axiosConfig);
    const items = res.data?.value || [];
    all.push(...items);
    next = res.data?.['@odata.nextLink'] || null;
  }
  return all;
}

// format recommendedd by aqs outlook
function mmddyyyy(d) {
  const m = (d.getUTCMonth() + 1).toString().padStart(2, '0');
  const day = d.getUTCDate().toString().padStart(2, '0');
  const y = d.getUTCFullYear();
  return `${m}/${day}/${y}`;
}

async function fetchAllPagesSearch(url, headers) {
  const all = await fetchAllPages(url, headers);
  all.sort((a, b) => new Date(b.receivedDateTime) - new Date(a.receivedDateTime));
  return all;
}

// Build AQS query string for Graph $search fallback (server-side)
function buildAqs({ from, toMailbox, subject, fromDate, toDate }) {
  const parts = [];
  if (isNonEmpty(from)) parts.push(`from:${from.trim()}`);
  if (isNonEmpty(toMailbox)) parts.push(`participants:${toMailbox.trim()}`);

  if (fromDate && toDate) {
    const f = mmddyyyy(new Date(fromDate));
    const t = mmddyyyy(new Date(toDate));
    parts.push(`received:${f}..${t}`);
  }

  if (isNonEmpty(subject)) {
    const toks = subject
      .toLowerCase().replace(/[^a-z0-9]+/g, ' ').trim().split(/\s+/).filter(Boolean);
    const uniq = Array.from(new Set(removeStopwords(toks)));
    if (uniq.length) parts.push('(' + uniq.map(t => `subject:${t}`).join(' AND ') + ')');
    // Optional exact-phrase bump (uncomment if you want it):
    // if (subject.includes(' ')) parts.push(`subject:"${subject.replace(/"/g, '\\"')}"`);
  }

  return parts.join(' AND ');
}

/* ------------ main worker ------------ */
function processAdvancedSearchOutlook(data) {
  return new Promise(async (resolve) => {
    try {
      const {
        token,
        filter,
        to,
        from, subject, fromDate, toDate
      } = data;

      const select =
        '$select=id,subject,from,receivedDateTime,body,ccRecipients,toRecipients,webLink';

      const baseHeaders = {
        'content-type': 'application/json',
        'authorization': `Bearer ${token}`
      };

      // $search requires this header
      const searchHeaders = {
        ...baseHeaders,
        'ConsistencyLevel': 'eventual'
      };

      let foundInTrash = false;
      let isMailFound = null;

      // 1) Inbox via $filter
      try {
        console.time(`fetch inbox ${to}`);
        const inboxUrl =
          `https://graph.microsoft.com/v1.0/users/${encodeURIComponent(to)}` +
          `/mailFolders/inbox/messages?$top=50&${select}${filter}`;
        const inbox = await fetchAllPages(inboxUrl, baseHeaders);
        console.timeEnd(`fetch inbox ${to}`);
        if (inbox.length) isMailFound = inbox;
      } catch (e) {
        console.log('Inbox fetch failed', to, e.response?.data || e.message);
      }

      // 2) Deleted Items via $filter
      if (!isMailFound) {
        try {
          console.time(`fetch deletedItems ${to}`);
          const delUrl =
            `https://graph.microsoft.com/v1.0/users/${encodeURIComponent(to)}` +
            `/mailFolders/deletedItems/messages?$top=50&${select}${filter}`;
          const deleted = await fetchAllPages(delUrl, baseHeaders);
          console.timeEnd(`fetch deletedItems ${to}`);
          if (deleted.length) {
            isMailFound = deleted;
            foundInTrash = true;
          }
        } catch (e) {
          console.log('Deleted Items fetch failed', to, e.response?.data || e.message);
        }
      }

      // 3) AQS $search fallback (server-only search)
      if (!isMailFound) {
        const aqs = buildAqs({ from, toMailbox: to, subject, fromDate, toDate });
        if (aqs) {
          const qs = `?$top=50&${select}&$search=${encodeURIComponent(`"${aqs}"`)}`;

          // 3a) inbox search
          try {
            console.time(`search inbox ${to}`);
            const sInboxUrl =
              `https://graph.microsoft.com/v1.0/users/${encodeURIComponent(to)}` +
              `/mailFolders/inbox/messages${qs}`;
            const sInbox = await fetchAllPagesSearch(sInboxUrl, searchHeaders);
            console.timeEnd(`search inbox ${to}`);
            if (sInbox.length) isMailFound = sInbox;
          } catch (e) {
            console.log('Search inbox failed', e.response?.status ? {
              status: e.response.status,
              code: e.response.data?.error?.code,
              message: e.response.data?.error?.message,
              requestId: e.response.headers?.['request-id']
            } : e.message);
          }

          // 3b) deleted items search
          if (!isMailFound) {
            try {
              console.time(`search deletedItems ${to}`);
              const sDelUrl =
                `https://graph.microsoft.com/v1.0/users/${encodeURIComponent(to)}` +
                `/mailFolders/deletedItems/messages${qs}`;
              const sDel = await fetchAllPagesSearch(sDelUrl, searchHeaders);
              console.timeEnd(`search deletedItems ${to}`);
              if (sDel.length) {
                isMailFound = sDel;
                foundInTrash = true;
              }
            } catch (e) {
              console.log('Search deletedItems failed', e.response?.status ? {
                status: e.response.status,
                code: e.response.data?.error?.code,
                message: e.response.data?.error?.message,
                requestId: e.response.headers?.['request-id']
              } : e.message);
            }
          }
        }
      }

      if (isMailFound?.length) {
        console.info('Found search result:', isMailFound[0].id, isMailFound[0].subject);
      }

      data.foundInTrash = foundInTrash;
      data.isMailFound  = isMailFound || null;
      resolve(data);
    } catch (err) {
      console.log(err);
      resolve(data);
    }
  });
}

module.exports = processAdvancedSearchOutlook;
