import React from "react";
import { Box, Typography, Modal, Button, List, ListItem, ListItemText, FormControlLabel, Checkbox } from "@mui/material";
import CloseSharpIcon from "@mui/icons-material/CloseSharp";
import styles from "../../styles/taConnectModal.module.scss";
import api from "../../utils/api";
import routes from "../../utils/routes";
import { toast } from "react-toastify";
export default function ConnectTaIntegrationModal(props) {
  const {
    handleIntegrationClose,
    newtaConnectopen
  } = props;

  const [agreed, setAgreed] = React.useState(false);

  const handleChange = (e) => setAgreed(e.target.checked);

  const terms = [
    {
      title: "Users and Groups Management",
      description:
        "All users and groups will be synced with Threatcop Admin and managed centrally, including add, edit, delete, and update actions.",
    },
    {
      title: "SSO Configuration",
      description:
        "Single Sign-On must be reconfigured for Threatcop Admin. Existing learner SSO will remain unaffected.",
    },
    {
      title: "Domain Management",
      description:
        "Your domains will be synced and managed via Threatcop Admin for consistent control.",
    },
    {
      title: "Application Access Routing",
      description:
        "All application logins (TSAT, TLMS, TDMARC, TPIR) will be routed through Threatcop Admin.",
    },
    {
      title: "MFA Setup Required",
      description:
        "Multi-Factor Authentication must be re-enabled for Threatcop Admin access.",
    },
    {
      title: "Domain Type Configuration",
      description:
        "You are required to configure domain types under the Domains section in Threatcop Admin.",
    },
    {
      title: "Integration Reconfiguration",
      description:
        "Existing Microsoft Azure and Google Workspace integrations must be reconfigured in Threatcop Admin.",
    },
  ];
    const changeConnectionStatusWithTa = async (status) => {
      try {
        const response = await api.post(routes.TA.CONNECT, { connectionStatus: status });
        const { success } = response.data;
        if (success) {
        toast.success(response.data.message);
        handleIntegrationClose();
        }
      } catch (error) {
        toast.error(error.response?.data?.message || "An error occurred while connecting to Threatcop Admin.");
        console.error("Error from TA-Connection:", error);
        handleIntegrationClose();
      }
      };

  return (
    <Modal
      open={newtaConnectopen}
      onClose={handleIntegrationClose}
      aria-labelledby="modal-modal-title"
      aria-describedby="modal-modal-description"
      className={styles.ConnectIntegrationModal}
    >
      <Box className={styles.mainmodalBody}>
        {/* Modal Header */}
        <Box className={styles.modalHeader}>
          <Typography variant="h5">Integration Consent Agreement</Typography>
          <Typography variant="body2">
            By proceeding, you agree to the followings
          </Typography>
          <CloseSharpIcon
            className={styles.closeIcon}
            onClick={handleIntegrationClose}
          />
        </Box>

        {/* Modal Body */}
        <Box className={styles.integratmodalBody}>
          <List className={styles.list}>
            {terms.map((term, index) => (
              <ListItem key={index} className={styles.listItem} alignItems="flex-start">
                <ListItemText
                  primary={
                    <Typography component="span" className={styles.listText}>
                      <strong>{index + 1}. {term.title}:</strong> {term.description}
                    </Typography>
                  }
                />
              </ListItem>
            ))}
          </List>

          {/* Modal Footer */}
          <Box className={styles.modalFooter}>
            <FormControlLabel
              control={<Checkbox checked={agreed} onChange={handleChange} />}
              label={
                <Typography variant="body2">
                  By clicking <strong>"I Agree"</strong>, you confirm your understanding and acceptance of the above conditions.
                </Typography>
              }
              className={styles.checkbox}
            />

            <Box>
              <Button
                variant="outlined"
                color="inherit"
                className={`${styles.btn} ${styles.btnSecondary}`}
                onClick={handleIntegrationClose}
              >
                Cancel
              </Button>

              <Button
                variant="contained"
                color="success"
                disabled={!agreed}
                className={styles.customButton}
                onClick={() => {
                  if (agreed) {
                    changeConnectionStatusWithTa(true);
                  }
                }}
              >
                Proceed Integration
              </Button>
            </Box>
          </Box>
        </Box>
      </Box>
    </Modal>
  );
}
