import React, { useMemo } from 'react';
import DOMPurify from 'dompurify';
import { Box, Typography } from '@mui/material';
import styles from '../../styles/home.module.scss';

const hasHTML = s => /<\/?[a-z!][\s\S]*>/i.test(s || '');

const looksLikeBase64 = s => /^[A-Za-z0-9+/]+={0,2}$/.test(s.replace(/\s+/g, ''));

const decodeBase64 = str => {
  try {
    const cleaned = str.replace(/^data:[^,]+,/, '').replace(/\s+/g, '');
    if (!looksLikeBase64(cleaned) || cleaned.length > 500000) return null;
    const decoded = atob(cleaned);
    return hasHTML(decoded) ? decoded : null;
  } catch {
    return null;
  }
};

const decodeQuotedPrintable = src => {
  const stripped = src.replace(/=(\r?\n)/g, '');
  const decoded = stripped.replace(/=([A-F0-9]{2})/gi, (_, h) =>
    String.fromCharCode(parseInt(h, 16))
  );
  return hasHTML(decoded) ? decoded : null;
};

const decodeEntities = str => {
  if (typeof window === 'undefined') return str;
  const t = document.createElement('textarea');
  t.innerHTML = str;
  return t.value;
};

const decodeBody = raw => {
  if (!raw) return '';
  const b64 = decodeBase64(raw);
  if (b64) return b64;
  if (/=[A-F0-9]{2}/i.test(raw) || /=\r?\n/.test(raw)) {
    const qp = decodeQuotedPrintable(raw);
    if (qp) return qp;
  }
  if (/&lt;\/?[a-z!]/i.test(raw)) return decodeEntities(raw);
  return raw;
};

export async function getServerSideProps() {
  return { props: {} };
}

export default function BodyTab({ bodyData = '', languages = [] }) {
  const decoded = useMemo(() => decodeBody(bodyData), [bodyData]);
  const isHTML = useMemo(() => hasHTML(decoded), [decoded]);

  const sanitizedHTML = isHTML
    ? (typeof window !== 'undefined'
        ? DOMPurify.sanitize(decoded)
        : decoded.replace(/<script[\s\S]*?>[\s\S]*?<\/script>/gi, '')
      )
    : decoded;

  const markup = isHTML ? { __html: sanitizedHTML } : null;

  return (
    <Box className={styles.AnalyticsReport}>
      <Box className={styles.HeaderDownloadBox}>
        <Typography component="p">
          Languages Detected:&nbsp;
          <b>{languages.length ? languages.join(', ') : '—'}</b>
        </Typography>
      </Box>
      <Box className={styles.MailBody}>
        {isHTML ? (
          <div dangerouslySetInnerHTML={markup} />
        ) : (
          <pre style={{ whiteSpace: 'pre-wrap', wordBreak: 'break-word' }}>
            {sanitizedHTML}
          </pre>
        )}
      </Box>
    </Box>
  );
}