/**
 * @license Copyright (c) 2003-2023, CKSource Holding sp. z o.o. All rights reserved.
 * For licensing, see LICENSE.md.
 */
import React from 'react';
import PropTypes, { type InferProps } from 'prop-types';
import type { EventInfo } from '@ckeditor/ckeditor5-utils';
import type { Editor, EditorConfig } from '@ckeditor/ckeditor5-core';
import { EditorWatchdog, ContextWatchdog } from '@ckeditor/ckeditor5-watchdog';
import type { WatchdogConfig } from '@ckeditor/ckeditor5-watchdog/src/watchdog';
export default class CKEditor<TEditor extends Editor> extends React.Component<Props<TEditor>, {}> {
    /**
     * Contains a promise that resolves when the editor destruction is finished.
     */
    private editorDestructionInProgress;
    /**
     * After mounting the editor, the variable will contain a reference to the created editor.
     * @see: https://ckeditor.com/docs/ckeditor5/latest/api/module_core_editor_editor-Editor.html
     */
    private domContainer;
    /**
     * An instance of EditorWatchdog or an instance of EditorWatchdog-like adapter for ContextWatchdog.
     * It holds the instance of the editor under `this.watchdog.editor` if `props.disableWatchdog` is set to false.
     */
    private watchdog;
    /**
     * Holds the instance of the editor if `props.disableWatchdog` is set to true.
     */
    private instance;
    constructor(props: Props<TEditor>);
    /**
     * An editor instance.
     */
    get editor(): Editor | null;
    /**
     * The CKEditor component should not be updated by React itself.
     * However, if the component identifier changes, the whole structure should be created once again.
     */
    shouldComponentUpdate(nextProps: Readonly<Props<TEditor>>): boolean;
    /**
     * Initialize the editor when the component is mounted.
     */
    componentDidMount(): Promise<void>;
    /**
     * Re-render the entire component once again. The old editor will be destroyed and the new one will be created.
     */
    componentDidUpdate(): Promise<void>;
    /**
     * Destroy the editor before unmounting the component.
     */
    componentWillUnmount(): Promise<void>;
    /**
     * Render a <div> element which will be replaced by CKEditor.
     */
    render(): React.ReactNode;
    /**
     * Initializes the editor by creating a proper watchdog and initializing it with the editor's configuration.
     */
    private _initializeEditor;
    /**
     * Creates an editor from the element and configuration.
     *
     * @param element The source element.
     * @param config CKEditor 5 editor configuration.
     */
    private _createEditor;
    /**
     * Destroys the editor by destroying the watchdog.
     */
    private _destroyEditor;
    /**
     * Returns true when the editor should be updated.
     *
     * @param nextProps React's properties.
     */
    private _shouldUpdateEditor;
    /**
     * Returns the editor configuration.
     */
    private _getConfig;
    static contextType: React.Context<ContextWatchdog<import("@ckeditor/ckeditor5-core").Context> | "contextWatchdog" | null>;
    static propTypes: {
        editor: PropTypes.Validator<{
            create(...args: any): Promise<any>;
        }>;
        data: PropTypes.Requireable<string>;
        config: PropTypes.Requireable<object>;
        disableWatchdog: PropTypes.Requireable<boolean>;
        watchdogConfig: PropTypes.Requireable<object>;
        onChange: PropTypes.Requireable<(...args: any[]) => any>;
        onReady: PropTypes.Requireable<(...args: any[]) => any>;
        onFocus: PropTypes.Requireable<(...args: any[]) => any>;
        onBlur: PropTypes.Requireable<(...args: any[]) => any>;
        onError: PropTypes.Requireable<(...args: any[]) => any>;
        disabled: PropTypes.Requireable<boolean>;
        id: PropTypes.Requireable<any>;
    };
    static _EditorWatchdog: typeof EditorWatchdog;
}
/**
 * TODO this is type space definition for props, the CKEditor.propTypes is a run-time props validation that should match.
 */
interface Props<TEditor extends Editor> extends InferProps<typeof CKEditor.propTypes> {
    editor: {
        create(...args: any): Promise<TEditor>;
    };
    config?: EditorConfig;
    watchdogConfig?: WatchdogConfig;
    disableWatchdog?: boolean;
    onReady?: (editor: TEditor) => void;
    onError?: (error: Error, details: ErrorDetails) => void;
    onChange?: (event: EventInfo, editor: TEditor) => void;
    onFocus?: (event: EventInfo, editor: TEditor) => void;
    onBlur?: (event: EventInfo, editor: TEditor) => void;
}
interface ErrorDetails {
    phase: 'initialization' | 'runtime';
    willEditorRestart?: boolean;
}
export {};
