import * as React from 'react';
import { useRouter } from 'next/router';
import { IoSearchOutline } from "react-icons/io5";
import {
    Box, Chip, Button, Typography, Grid,
    FormControl, Select, MenuItem, TextField,
    TableRow, TableHead, TableContainer, TableCell, TableBody, Table, Stack,
    Checkbox
} from '@mui/material';

import { TfiSearch } from "react-icons/tfi";
import { TfiReload } from "react-icons/tfi";
import "react-datepicker/dist/react-datepicker.css";

import api from '../../utils/api';
import routes from '../../utils/routes';
import { apiError } from '../../utils/error';
import styles from '../../styles/home.module.scss';
import Loader from '../../components/loader/loader';
import { getDatewithoutTime } from '../../utils/common';
import Pagination from '../../components/globlePagination';
import AdvanceSearchModal from '../../components/advance-search-modal/advance-search-modal';
import { useFeatures } from '../../utils/features';
import { PiTrash } from "react-icons/pi";
import { toast } from 'react-toastify';
import swal from 'sweetalert'


export default function AdvanceSearch(props) {
    const router = useRouter();
    const { canAccessFeature, loadFeatures, areFeaturesLoaded } = useFeatures()
    const [showDomain, setShowDomain] = React.useState(false)
    const [page, setPage] = React.useState(1);
    const [count, setCount] = React.useState(1);
    const [limit, setLimit] = React.useState(10);
    const [search, setSearch] = React.useState('');
    const [loader, setLoader] = React.useState(false);
    const [open, setOpen] = React.useState(false);
    const handleOpen = () => setOpen(true);
    const handleClose = () => setOpen(false);
    const [advanceData, setAdvanceData] = React.useState([]);
    const [domains, setDomains] = React.useState([])
    const [selectedDomain, setSelectedDomain] = React.useState('-1')
    const [selectedItems, setSelectedItems] = React.useState(new Set());
    const [selectAllCurrentPage, setSelectAllCurrentPage] = React.useState(false);

    const handleSelectItem = (itemId) => {
        const newSelectedItems = new Set(selectedItems);
        if (newSelectedItems.has(itemId)) {
            newSelectedItems.delete(itemId);
        } else {
            newSelectedItems.add(itemId);
        }
        setSelectedItems(newSelectedItems);

        // Update select all checkbox state
        const currentPageIds = advanceData.map(item => item._id);
        const allCurrentPageSelected = currentPageIds.every(id => newSelectedItems.has(id));
        setSelectAllCurrentPage(allCurrentPageSelected);
    };

    // Handler for select all current page
    const handleSelectAllCurrentPage = (event) => {
        const isChecked = event.target.checked;
        const newSelectedItems = new Set(selectedItems);
        const currentPageIds = advanceData.map(item => item._id);

        if (isChecked) {
            currentPageIds.forEach(id => newSelectedItems.add(id));
        } else {
            currentPageIds.forEach(id => newSelectedItems.delete(id));
        }

        setSelectedItems(newSelectedItems);
        setSelectAllCurrentPage(isChecked);
    };


    const handleDeleteSelectedAdvanceSearch = async () => {
        if (selectedItems.size === 0) return;

        setLoader(true);

        const willDelete = await swal({
            title: "Are you sure?",
            text: "Are you sure you want to Delete these entries?",
            icon: "warning",
            dangerMode: true,
            closeOnEsc: true,
            buttons: {
                cancel: true,
                confirm: true,
            },
        });
        if (willDelete) {
            api.post(routes.ADVANCE_SEARCH.REMOVE, {
                ids: Array.from(selectedItems)
            })
                .then(res => res.data)
                .then(data => {
                    setLoader(false);
                    toast.success(data.message)
                    setSelectedItems(new Set());
                    setSelectAllCurrentPage(false);
                    loadAdvanceSearchList();
                })
                .catch(err => {
                    setLoader(false);
                    apiError(err);
                });

        }


    }

    // Reset selection when page changes
    React.useEffect(() => {
        setSelectAllCurrentPage(false);
    }, [page]);

    const handleSelectedDomain = (event) => {
        const domainId = event.target.value
        setSelectedDomain(domainId);
        setPage(1)
    };

    const handlePageChange = (event, value) => {
        setPage(value);
    };

    const loadAdvanceSearchList = () => {
        setLoader(true)
        api.get(routes.ADVANCE_SEARCH.LIST, {
            params: {
                domainId: selectedDomain && selectedDomain !== '-1' ? selectedDomain : null, page, limit, search
            }
        })
            .then(res => res.data)
            .then(data => {
                setLoader(false)
                setCount(data.count);
                setAdvanceData(data.results);
            })
            .catch(err => {
                setLoader(false)
                apiError(err);
            })
    }

    const reportedMail = (reportId) => {
        router.push(`/advance-search/${reportId}`)
    }

    const loadDomainList = () => {
        api.get(routes.DOMAIN.LIST)
            .then(res => res.data)
            .then((data) => {
                if (data.domains.length && data.domains.length > 0) {
                    const domain = data.domains.filter((element) => {
                        if (element.isCredential !== false && element.domainTypeForOutlook != 'EXCHANGE') return element
                    })
                    setDomains(domain)
                    setShowDomain(true)
                }
            })
            .catch((error) => {
                if (!(error?.response?.status === 403)) apiError(error)
                else {
                    setShowDomain(false)
                }
            })
    }

    React.useEffect(() => {
        loadFeatures()
    }, [])

    React.useEffect(() => {
        loadDomainList()
    }, [])

    React.useEffect(() => {
        loadAdvanceSearchList();
    }, [page, limit, count, search, selectedDomain])

    React.useEffect(() => {
        setPage(1)
    }, [search])


    React.useEffect(() => {
        if (canAccessFeature("TPIR-ADVS")) {
            loadAdvanceSearchList()
        }
        else {
            if (areFeaturesLoaded) router.push('/feature-unavailable')
        }
    }, [areFeaturesLoaded])

    const getTimeDifference = (date1, date2) => {
        var startDate = new Date(date1);
        var endDate = new Date(date2);
        var seconds = (endDate.getTime() - startDate.getTime()) / 1000;
        return isNaN(seconds) ? '--' : Math.ceil(seconds);
    }

    return (
        !areFeaturesLoaded ? <box>Loading</box> :

            <Box className={styles.contentbar}>
                <Box className={styles.containerbar}>
                    <Box className={styles.breadcrumbBar}>
                        <Grid container direction="row" justifyContent="center" alignItems="center" >
                            <Grid item xs={4}>
                                <Typography component='h4' variant='h4' className={styles.page_title}>Advance Search</Typography>
                            </Grid>
                            <Grid item xs={8} align="right">
                                <Button variant="contained" className={`${styles.btn} ${styles.btnPrimary}`} onClick={loadAdvanceSearchList} ><TfiReload className={styles.icon_position} />Refresh</Button>
                                <Button variant="contained" className={`${styles.btn} ${styles.btnPrimary}`} onClick={handleOpen}><TfiSearch className={styles.icon_position} /> New Search</Button>
                            </Grid>
                        </Grid>
                    </Box>

                    <Box className={styles.filterWrapper}>
                        <Grid container direction="row" justifyContent="center" alignItems="center" >
                            <Grid item xs={12} md={4} align="left" position="relative">
                                <Box className={styles.filterSearchBar}>
                                    <TextField
                                        className={styles.searchBar}
                                        placeholder='Search'
                                        size="small"
                                        fullWidth
                                        value={search}
                                        onChange={e => setSearch(e.target.value)}
                                    />
                                    <Typography component="span" className={styles.search_icon} ><IoSearchOutline /> </Typography>
                                </Box>
                            </Grid>

                            <Grid item xs={12} md={8}>
                                <Box className={styles.reportFitter}>
                                    {showDomain &&
                                        <Box>
                                            <FormControl >
                                                <Select
                                                    className={`${styles.btn} ${styles.btnPrimaryDrop}`}
                                                    labelId="demo-simple-select-autowidth-label"
                                                    id="demo-simple-select-autowidth"
                                                    value={selectedDomain}
                                                    onChange={handleSelectedDomain}
                                                    autoWidth
                                                    displayEmpty
                                                    sx={{ boxShadow: 'none', '.MuiOutlinedInput-notchedOutline': { borderRadius: '0px', paddginLeft: '20px' } }}
                                                >
                                                    <MenuItem value="-1">Select Domain</MenuItem>
                                                    {domains.map((data) => <MenuItem value={data._id} key={data._id}>{data.domainName} ({data.domainType}) </MenuItem>)}
                                                </Select>
                                            </FormControl>
                                            <FormControl>
                                                <Select
                                                    className={`${styles.btn} ${styles.btnPrimaryDrop}`}
                                                    labelId="demo-simple-select-autowidth-label"
                                                    id="demo-simple-select-autowidth"
                                                    value={limit}
                                                    onChange={e => setLimit(e.target.value)}
                                                    autoWidth
                                                    displayEmpty
                                                    sx={{ boxShadow: 'none', '.MuiOutlinedInput-notchedOutline': { borderRadius: '0px', paddginLeft: '20px' } }}
                                                >
                                                    <MenuItem value="10">10 </MenuItem>
                                                    <MenuItem value="25">25 </MenuItem>
                                                    <MenuItem value="50">50 </MenuItem>
                                                    <MenuItem value="100">100 </MenuItem>
                                                </Select>
                                            </FormControl>
                                        </Box>
                                    }
                                </Box>
                            </Grid>

                        </Grid>
                    </Box>


                    <Box className={styles.contentbarWrap}>
                        {selectedItems.size > 0 && (
                            <Box sx={{ mb: 2, display: 'flex', gap: 1 }}>
                                <Typography variant="body2" sx={{ alignSelf: 'center', paddingRight: '10px', borderRight: '1px solid #212529', fontWeight: '500' }}>
                                    {selectedItems.size} selected
                                </Typography>
                                <Button variant="outlined" className={styles.btnDangerLight} size="small" onClick={() => handleDeleteSelectedAdvanceSearch()}>
                                    <PiTrash style={{ marginRight: '5px' }} /> Delete
                                </Button>
                            </Box>
                        )}
                        <TableContainer className={styles.tableContainer}>

                            <Table sx={{ minWidth: 650 }} aria-label="simple table" className={styles.ReportedMailTable}>
                                <TableHead className={styles.thead_background}>
                                    <TableRow className={styles.advanceSearchTable}>
                                        <TableCell padding="checkbox">
                                            <Checkbox
                                                size='small'
                                                checked={selectAllCurrentPage}
                                                onChange={handleSelectAllCurrentPage}
                                                style={{ padding: '0px' }}
                                            />
                                        </TableCell>
                                        <TableCell>From</TableCell>
                                        <TableCell>To</TableCell>
                                        <TableCell>Sender</TableCell>
                                        <TableCell>Receiver</TableCell>
                                        <TableCell>Subject</TableCell>
                                        <TableCell>Time Taken</TableCell>
                                        <TableCell> Email Count</TableCell>
                                        <TableCell> Status</TableCell>
                                    </TableRow>
                                </TableHead>
                                {advanceData && advanceData.length > 0 &&
                                    <TableBody className={styles.reportedMail}>
                                        {advanceData.map((data, index) => (
                                            <TableRow key={index} sx={{ '&:last-child td, &:last-child th': { border: 0 } }} onClick={() => reportedMail(data._id)}>
                                                <TableCell padding="checkbox" onClick={(e) => e.stopPropagation()}>
                                                    <Checkbox
                                                        size='small'
                                                        checked={selectedItems.has(data._id)}
                                                        style={{ padding: '0px' }}
                                                        onChange={() => handleSelectItem(data._id)}
                                                    />
                                                </TableCell>
                                                <TableCell component="td" scope="row">{getDatewithoutTime(data.fromDate)}</TableCell>
                                                <TableCell component="td" scope="row">{getDatewithoutTime(data.toDate)}</TableCell>
                                                <TableCell component="td" scope="row">{data.from}</TableCell>
                                                <TableCell component="td" scope="row">{data.to ? data.to : '--'}</TableCell>
                                                <TableCell component="td" scope="row">{data.subject}</TableCell>

                                                <TableCell component="td" scope="row" sx={{ textAlign: "center", fontWeight: "700" }}>{
                                                    getTimeDifference(data.syncStartAt, data.syncEndAt)
                                                }s</TableCell>
                                                <TableCell component="td" scope="row" sx={{ textAlign: "center", fontWeight: "700" }}>{data.emailCount}</TableCell>
                                                <TableCell component="td" scope="row">
                                                    <Box className={styles.mddelete}>
                                                        <Stack direction="row" spacing={1}>
                                                            {data.isProcessing ?
                                                                <Chip label="Processing" variant="outlined" size="small" color="primary" /> :
                                                                <Chip label="Completed" variant="outlined" size="small" color="success" />
                                                            }
                                                        </Stack>
                                                    </Box>
                                                </TableCell>
                                            </TableRow>
                                        ))}
                                    </TableBody>
                                }
                            </Table>
                            {advanceData && advanceData.length > 0 ?
                                <Pagination
                                    count={count}
                                    page={page}
                                    limit={limit}
                                    handlePageChange={handlePageChange}
                                ></Pagination> :
                                <>{!loader ?
                                    <Box className={`${styles.noRecordMail}`}>
                                        <Typography component="p">No Records Found!</Typography>
                                    </Box> :
                                    <Box className={`${styles.noRecordMail}`}>
                                        <Typography component="p" style={{ color: "green" }}>Loading...</Typography>
                                    </Box>
                                }
                                </>
                            }
                            {/* { loader && <Loader />} */}
                        </TableContainer>
                    </Box>

                    <AdvanceSearchModal
                        handleClose={handleClose}
                        open={open}
                        domains={domains}
                        loadAdvanceSearchList={loadAdvanceSearchList}
                    ></AdvanceSearchModal>
                </Box>
            </Box >
    )
}