import * as React from "react";
import Image from "next/image";
import { useEffect } from "react";
import { useRouter } from "next/router";
import { LoadingButton } from "@mui/lab";
import { Box, TextField, Typography, InputLabel, FormControl, MenuItem, Select} from "@mui/material";

import { toast } from 'react-toastify';
import { ToastContainer } from 'react-toastify';
import 'react-toastify/dist/ReactToastify.css';

import api from "../utils/api";
import routes from "../utils/routes";
import { apiError } from "../utils/error";
import logo from '../assets/logo/Tpirlogin.png';
import styles from "../styles/auth.module.scss";
import { validateEmail } from "../utils/common";

export default function ForgotPassword(props) {
	const router = useRouter();
	const [forgotData, setForgotEmail] = React.useState({
		email: "",
		companyId: "-1",
		type: "root",
		callAPI: false,
	});

	const [loading, setLoading] = React.useState(false);

	const [emailError, setEmailError] = React.useState({ status: false, message: ""});
	const [companyError, setCompanyError] = React.useState({ status: false, message: ""});

	const [isShowCompany, setIsShowCompany] = React.useState(false);
	const [isShowSendButton, setIsShowSendButton] = React.useState(false);
	const [companies, setCompanies] = React.useState([]);

	const [messages, setMessage] = React.useState("");
	const [onSuccess, setSuccess] = React.useState(false);

	const handleChange = (e) => {
		setForgotEmail({ ...forgotData, [e.target.name]: e.target.value });
	};

	const handleCompanyChange = (e) => {
		if (e.target.value !== "-1") {
			let target = e.target.value.split("#");
			const type = target[1];
			const companyId = target[0];
			setForgotEmail({ ...forgotData, companyId: companyId, type: type });
			setCompanyError((prevState)=> ({...prevState, status: false, message: ''}))
		} 
		else {
			setForgotEmail({ ...forgotData, companyId: "-1" });
			setCompanyError((prevState)=> ({...prevState, status: true, message: 'Please Select a company!'}))
		}
	};

	useEffect(() => {
		const { callAPI } = forgotData;
		if (callAPI) {
			sendMail();
		}
	}, [forgotData.callAPI]);

	const sendMail = () => {
		const { email, companyId, type } = forgotData;
		if ( companyId != -1 ){
			setLoading(true);
			api.post(routes.AUTH.FORGOT_PASSWORD, {
				email,
				companyId: companyId ? companyId : selectedCompanyId,
				type
			})
			.then((res) => res.data)
			.then((data) => {
				setLoading(false);
				setMessage("Password Reset Link Has Been sent to your registered mail id!");
				setSuccess(true);
				setTimeout(() => {
				router.push("/login");
				}, 3000);
			})
			.catch((error) => {
				setLoading(false);
				apiError(error);
			});
		} else {
			setCompanyError({...companyError, status: true, message: 'Please select a company!'})
		}
	};

	const checkEmail = () => {
		if ( forgotData.email && forgotData.email.length && validateEmail(forgotData.email) ) {
			setLoading(true);
			api.post(routes.AUTH.CHECK_EMAIL, { email: forgotData.email })
			.then((res) => res.data)
			.then((data) => {
				setLoading(false);
				if (data.companies.length === 1) {
					setForgotEmail((prevState) => ({...prevState, companyId: data.companies[0]._id, 
					type: data.companies[0].type ? data.companies[0].type : "root", callAPI: true }));
				} 
				else {
					setIsShowCompany(true);
					setCompanies(data.companies);
					setIsShowSendButton(true);
				}
				})
			.catch((err) => {
				setLoading(false);
				apiError(err);
			});
		} else {
			setEmailError({ status: true, message: "Please Enter a Valid Email Id" });
		}
	};

	const getCompanyIdValue = (companyId, type) => {
		let value = companyId;
		for (let i = 0; i < companies.length; i++) {
			let comType = companies[i].type ? companies[i].type : "root";
			if (companies[i]._id === companyId && comType === type) {
				value = `${companyId}#${companies[i].type ? companies[i].type : "root"}`;
				break;
			}
		}
		return value;
	};

	const handleEnterButton = (e) => {
        switch(e.target.name){
            case 'email': if(e.key === "Enter") checkEmail()
        }
      };
	  
	return (
		<>
		<Box className={styles.loginContainer}>
			<Box className={styles.logoHeader}>
				<Box className={styles.imgBox}>
					<Image src={logo} alt="no image"/>
				</Box>
			</Box>
			{!onSuccess? (
				<>
					<Box className={styles.authBox}>
						<Typography component="h4" variant="h3">
							Forgot Password
						</Typography>
						<Box className={styles.common_box_input}>
							<InputLabel sx={{marginBottom:"7px"}}>Please enter your registerd email id</InputLabel>
							<TextField
								className={styles.input}
								value={forgotData.email}
								placeholder='Email Address'
								name="email"
								size="small"
								type="text"
								disabled={isShowCompany}
								onKeyDown={handleEnterButton}
								onChange={handleChange}
								variant="outlined"
								fullWidth
								error={emailError.status}
								helperText={emailError.message}
								/>
						</Box>
						{isShowCompany && (
						<Box className={styles.common_box_input}>
							<InputLabel>Select Login Company</InputLabel>
							<FormControl fullWidth size="small" className={styles.input}>
								<Select
									id="demo-simple-select"
									value={getCompanyIdValue(
									forgotData.companyId,
									forgotData.type
									)}
									onChange={handleCompanyChange}
									className={styles.user_select}
									name="Company"
								>
									<MenuItem value="-1">Select Company</MenuItem>
									{companies.map((data, index) => {
									return (
										<MenuItem
										key={index}
										value={`${data._id}#${
											data.type ? data.type : "root"
										}`}
										>
										{`${data.name} (${data.type ? data.type : "root"})`}
										</MenuItem>
									);
									})}
								</Select>
							</FormControl>
						{companyError.status && <p style={{color: 'red', fontSize: '12px'}}>{companyError.message}</p>}
						</Box>
						)}
						{isShowSendButton ? 
							<LoadingButton
								disabled={loading}
								loading={loading}
								loadingPosition="start"
								variant="contained"
								className={styles.loginBtn}
								onClick={sendMail}
							> Send Reset Link </LoadingButton>:
							<LoadingButton
								disabled={loading}
								loading={loading}
								loadingPosition="start"
								variant="contained"
								className={styles.loginBtn}
								onClick={checkEmail}
							> Next </LoadingButton>
						}
					</Box>
				</>
			) : (
				<p> {messages} </p>
			)}
		</Box>
        <ToastContainer />
		</>
	);
}
