import React, { useState } from 'react';
import Link from 'next/link';
import Image from 'next/image';
import Cookies from 'js-cookie';
import { useRouter } from 'next/router';
import { LoadingButton } from '@mui/lab';
import { Box, TextField, Typography, InputLabel, FormControl, MenuItem, Select, InputAdornment, } from '@mui/material';
import { Visibility, VisibilityOff } from '@mui/icons-material';

import { toast } from 'react-toastify';
import { ToastContainer } from 'react-toastify';
import 'react-toastify/dist/ReactToastify.css';

import api from '../utils/api';
import routes from '../utils/routes';
import { apiError } from '../utils/error';
import logo from '../assets/logo/Tpirlogin.png';
import styles from '../styles/auth.module.scss';
import { AppContext } from '../components/appcontext';

export default function Login() {

    const router = useRouter();
    const [authData, setAuthData] = useState({
        email: '',
        type: '',
        otp: '',
        password: '',
        companyId: "-1",
        type: 'root'
    })
    const [authDataError, setAuthDataError] = useState({
        email: { status: false, message: '' },
        type: { status: false, message: '' },
        otp: { status: false, message: '' },
        password: { status: false, message: '' },
        companyId: { status: false, message: '' },
    })
    const [loading, setLoading] = React.useState(false);
    const [companies, setCompanies] = useState([]);
    const [isShowOTP, setShowOTP] = useState(false);
    const [isShowOTPMail, setShowOTPMail] = useState(false);
    const [isShowCompany, setShowCompany] = useState(false);
    const [isShowPassword, setShowPassword] = useState(false);
    const [isShowOTPMailTime, setShowOTPMailTime] = useState(30);
    const [selectedCompany, setSelectedCompany] = React.useState('-1')
    const [isShowForgotPassword, setShowForgotPassword] = useState(false);
    const [showPassword, setEyeShowPassword] = useState(false);
    const { domains,setDomains, setDomainChange } = React.useContext(AppContext);



    const handleTogglePassword = () => {
        setEyeShowPassword((prevState) => !prevState);
    };
    const handleChange = (e) => {
        setAuthData({ ...authData, [e.target.name]: e.target.value });
        setAuthDataError({ ...authDataError, [e.target.name]: { status: false, message: '' } });
    };

    const ValidateEmail = email => {
        let re = /^(([^<>()\[\]\\.,;:\s@"]+(\.[^<>()\[\]\\.,;:\s@"]+)*)|(".+"))@((\[[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\])|(([a-zA-Z\-0-9]+\.)+[a-zA-Z]{2,}))$/;
        return re.test(String(email).toLowerCase());
    };  
    
   
    

    const validateLogin = () => {
        const { companyId, password,otp } = authData;   
        let isError = false;
        if( isShowOTP || isShowOTPMail ){
            if (!otp || !otp.length) {
                isError = true;
                setAuthDataError({ ...authDataError, otp: { status: true, message: 'Please Enter otp' } });
            }
        }
        if (!password || !password.length) {
            isError = true;
            setAuthDataError({ ...authDataError, password: { status: true, message: 'Please Enter Password' } });
        }
        if (!companyId || !companyId.length || companyId === '-1') {
            isError = true;
            setAuthDataError({ ...authDataError, companyId: { status: true, message: 'Please Select Login Company' } });
        }
        return isError? !isError: authData;
    }

    const login = () => {
        const data = validateLogin();
        if (data) {
            setLoading(true);
            api.post(routes.AUTH.LOG_IN, data)
            .then(res => res.data)
            .then(async data => {
                setLoading(false);
                let expiredTime = new Date(new Date().setMinutes(new Date().getMinutes()+180));
                if(data.isURL) Cookies.set('logo', data.url, { expires: expiredTime,  })
                Cookies.set('token', data.token, { expires: expiredTime,  secure: true })
                Cookies.set('username', data.name, { expires: expiredTime  })
                Cookies.set('email', data.email, { expires: expiredTime  })
                Cookies.set('isTacEnabled', data.isTacEnabled)
                //api.defaults.headers.Authorization = `Bearer ${data.token}`
                setDomainChange(prevState => !prevState)
                router.push('/home')
                if(!data?.isTacEnabled) sessionStorage.setItem('firstLogin', 'true');
            })
            .catch(err => {
                setLoading(false);
                apiError(err);
            })
        }
    }

    React.useEffect(() => {
        checkCompany();
    }, [authData.companyId])

    const [loginViaTac, setLoginViaTac] = React.useState({
        show: false,
        companyName:'',
        alias:''
    })

    const handleCompanyChange = async(e) => {
        if(e.target.value !== '-1'){
            let target = (e.target.value).split('#');
            const type = target[1];
            const companyId = target[0];
            setSelectedCompany(e.target.value)
            setAuthData({ ...authData, companyId, type });
            setAuthDataError({ ...authDataError, companyId: { status: false, message: 'Please Select Login Company' } });
        } else {
            setShowOTP(false);
            setShowPassword(false);
            setAuthData({ ...authData, companyId: '-1' });
            setAuthDataError({ ...authDataError, companyId: { status: true, message: 'Please Select Login Company' } });
        }
    }

    const checkCompany = () => {
        const { companyId, type, email } = authData;
        if(companyId !== '-1'){
            setLoading(true);
            api.post(routes.AUTH.CHECK_COMPANY, {
                email, type, companyId
            })
            .then(res => res.data)
            .then(data => {
                if(data?.showLoginViaTa?.show){
                    setLoginViaTac(data?.showLoginViaTa)
                    setLoading(false);
                }
                else{
                    if (data.isOtp) {
                        setShowOTP(true)
                        if(data.otpType == 'MAIL'){
                            setShowOTPMail(true);
                        }
                        toast.success(data.message);
                    }
                    setLoading(false);
                    setShowPassword(true);
                }
            })
            .catch(err => {
                setLoading(false);
                apiError(err);
            })
        }
    }

    const resentOTP = () => {
        const { companyId, type, email } = authData;
        if(companyId !== '-1'){
            setLoading(true);
            api.post(routes.AUTH.RESEND_OTP, {
                email, type, companyId
            })
            .then(res => res.data)
            .then(data => {
                setLoading(false);
                setShowPassword(true);
                setShowOTPMailTime(30);
                toast.success("OTP has been sent to your emailId")
            })
            .catch(err => {
                setLoading(false);
                apiError(err);
            })
        }
    }

    const checkEmail = () => {
        const { email } = authData;
        if (email && email.length) {
            if (ValidateEmail(email)) {
                setLoading(true);
                api.post(routes.AUTH.CHECK_EMAIL, { email })
                .then(res => res.data)
                .then(data => {
                    if (data.companies.length === 1) {
                        setAuthData({ ...authData,
                            companyId: data.companies[0]._id,
                            type: data.companies[0].type? data.companies[0].type: 'root'
                        });
                    } else {
                        setShowCompany(true);
                    }
                    setLoading(false);
                    setCompanies(data.companies)
                    setShowForgotPassword(true);
                })
                .catch(err => {
                    setLoading(false);
                    apiError(err);
                })
            } else {
                setAuthDataError({ ...authDataError, email: { status: true, message: 'Please Enter a valid EmailId' } });
            }
        } else {
            setAuthDataError({ ...authDataError, email: { status: true, message: 'Please Enter EmailId' } });
        }
    }

    const routeToTa = () =>{
        let companyName = encodeURIComponent(loginViaTac?.companyName)
        let alias = encodeURIComponent(loginViaTac?.alias)
        let email = encodeURIComponent(authData?.email)
        router.push(`${process.env.NEXT_PUBLIC_TAC_URL}/login?companyName=${companyName}&alias=${alias}&email=${email}`);
    }
    
    const getCompanyIdValue = (companyId) => {
        let value = companyId;
        for(let i=0; i<companies.length; i++){
            if(companies[i]._id == companyId){
                value = `${companyId}#${companies[i].type?companies[i].type:'root'}`
            }
        }
        return value;
    }

    const handleEnterButton = (e) => {
        switch(e.target.name){
            case 'email':if(e.key === "Enter") checkEmail()
            case 'password': if(e.key === "Enter") login()
            case 'otp': if(e.key === "Enter") login()
        }
      };
      
    React.useEffect(() => {
        if( isShowOTP && isShowOTPMail && isShowOTPMailTime > 0 ){
            setTimeout(() => {
                setShowOTPMailTime(isShowOTPMailTime-1)
            }, 1000)
        }
    }, [isShowOTP, isShowOTPMail, isShowOTPMailTime])

    React.useEffect(() => {
        Cookies.remove('logo');
        Cookies.remove('token');
        Cookies.remove('username');
    }, [])
    return (
        <>
        <Box className={styles.loginContainer}>
            <Box className={styles.logoHeader}>
                <Box className={styles.imgBox}>
                    <Image src={logo} />
                </Box>
            </Box>
            <Box className={styles.authBox}>
                <Typography component='h4' variant='h4'>Log in to your account</Typography>

                <Box className={styles.common_box_input}>
                    <label>Email Address</label>
                    <TextField 
                        disabled={isShowForgotPassword}
                        error={authDataError.email.status}
                        helperText={authDataError.email.message}
                        value={authData.email}
                        name="email"
                        placeholder='Email Address'
                        required
                        size="small"
                        type="text"
                        id="outlined-start-adornment"
                        onChange={handleChange}
                        onKeyDown={handleEnterButton}
                        sx={{marginTop:"7px" , fontSize:"14px"}}
                        variant="outlined" fullWidth
                    className={styles.InputField}></TextField>
                </Box>
                {isShowCompany &&
                    <Box className={styles.common_box_input}>
                        <InputLabel>Select Login Company</InputLabel>
                        <FormControl fullWidth size='small' className={styles.InputField}  sx={{marginTop:"7px" , fontSize:"14px"}}>
                            <Select
                                id="demo-simple-select"
                                value={selectedCompany}
                                onChange={handleCompanyChange}
                                className={styles.user_select}
                                size="small"
                                name="companyId"
                                sx={{ color:"#838b91", fontSize:"14px", padding:"3px 0px" }}>
                                <MenuItem value="-1" disabled>Select Company</MenuItem>
                                {companies.map((data, index) => {
                                    return <MenuItem key={index} value={`${data._id}#${data.type?data.type:'root'}`}>{`${data.name} (${data.type ? data.type : "root"})`}</MenuItem>
                                })}
                            </Select>
                            {authDataError.companyId.status && <p style={{color: 'red', fontSize: '14px'}}>{authDataError.companyId.message}</p>}
                        </FormControl>
                    </Box>
                } {isShowPassword &&
                    <Box className={styles.common_box_input}>
                        <label>Password</label>
                        <TextField
                            error={authDataError.password.status}
                            helperText={authDataError.password.message}
                            placeholder='Password'
                            className={styles.InputField}
                            value={authData.password}
                            name="password"
                            size='small'
                            sx={{marginTop:"7px" , fontSize:"14px"}}
                            type={showPassword ? 'text' : 'password'}
                            onKeyDown={handleEnterButton}
                            onChange={handleChange}
                            variant="outlined" fullWidth 
                            InputProps={{
                                endAdornment: (
                                <InputAdornment position="end">
                                    <span onClick={handleTogglePassword} className={styles.eyeIcon}>
                                      {showPassword ? <VisibilityOff /> : <Visibility />}
                                    </span>
                                </InputAdornment>
                                ),
                            }}
                            />
                           
                    </Box>
                } {isShowOTP &&
                    <Box className={`${styles.common_box_input}, ${styles.optTimer}`}>
                        <TextField 
                            error={authDataError.otp.status}
                            helperText={authDataError.otp.message}
                            className={styles.input}
                            value={authData.otp} 
                            name="otp" size='small' 
                            type="password"
                            onKeyDown={handleEnterButton}
                            onChange={handleChange}
                            placeholder='One Time Password'
                            variant="outlined" fullWidth 
                        />
                        {isShowOTP && isShowOTPMail && isShowOTPMailTime > 0 && <span className={styles.showOtpTimer}>00:{isShowOTPMailTime<10?'0':''}{isShowOTPMailTime}</span> }
                    </Box>
                }

                <Box className={styles.forgetBtn}>
                    <Box className={styles.ResetBtn}>
                      { isShowOTP && isShowOTPMail && isShowOTPMailTime <=0 && <a className={styles.resend_otp} onClick={resentOTP}>Resend</a>}
                    </Box>
                    <Box className={styles.ForgetBox}>
                    {isShowForgotPassword &&
                        (loginViaTac?.show ? (
                        <Link href={`${process.env.NEXT_PUBLIC_TAC_URL}/forgot-password?email=${encodeURIComponent(authData?.email || '')}`} className={styles.forget_password}>
                            Forget Your Password
                        </Link>
                        ) : (
                        <Link href="/forgot-password" className={styles.forget_password} >
                            Forget Your Password
                        </Link>
                        ))}
                    </Box>
                    
                </Box>

                {!loginViaTac.show ? (
                !isShowForgotPassword ? (
                    <LoadingButton
                    disabled={loading}
                    loading={loading}
                    loadingPosition="start"
                    variant="contained"
                    className={styles.loginBtn}
                    onClick={checkEmail}
                    >
                    Next
                    </LoadingButton>
                ) : (
                    <LoadingButton
                    disabled={loading}
                    loading={loading}
                    loadingPosition="start"
                    variant="contained"
                    className={styles.loginBtn}
                    onClick={login}
                    >
                    Login
                    </LoadingButton>
                )
                ) : null}

                {loginViaTac && loginViaTac.show && 
                    <LoadingButton 
                        disabled={loading} 
                        loading={loading} 
                        loadingPosition="start" 
                        variant="contained" 
                        className={styles.loginBtn} 
                        onClick={routeToTa}
                    >Login via TA</LoadingButton>
                }
            </Box>
            <Typography component="p" variant="body1" className={styles.auth_footer}> Don't have an account?{" "} <Link href={`${process.env.NEXT_PUBLIC_TAC_URL}/register`}> Click here </Link> </Typography>
            <Typography component='p' variant='p' className={styles.auth_footer}>If you need help, <Link href="mailto:support@tpir.threatcop.com">contact support</Link>.</Typography>
        </Box>
        <ToastContainer />
        </>

    )
}