import { IoCheckmark } from "react-icons/io5";
import React, { useState } from "react";
import Cookies from "js-cookie";
import PropTypes from "prop-types";
import { useRouter } from "next/router";
import Link from "next/link";

import {
  Box,
  Button,
  Typography,
  Grid,
  InputLabel,
  Switch,
  TextField,
  Tab,
  Tabs,
  Modal,
  Select,
  MenuItem,
  TableContainer,
  Table,
  TableHead,
  TableRow,
  TableCell,
  TableBody,
  FormControl,
  Tooltip,
  Stack,
} from "@mui/material";
import CloseSharpIcon from "@mui/icons-material/CloseSharp";
import { TabContext, TabList } from "@mui/lab";
import { LoadingButton } from "@mui/lab";
import { IoInformation } from "react-icons/io5";
import { IoCheckmarkOutline } from "react-icons/io5";
import { TfiCommentAlt, TfiLink } from "react-icons/tfi";
import { toast } from "react-toastify";
import { TfiLock } from "react-icons/tfi";
import { SiGmail } from "react-icons/si";
import { TfiPlus } from "react-icons/tfi";
import { TfiMinus } from "react-icons/tfi";
import { TfiMobile } from "react-icons/tfi";
import { TiEdit } from "react-icons/ti";
import Radio from "@mui/material/Radio";
import { TfiEmail } from "react-icons/tfi";
import FormControlLabel from "@mui/material/FormControlLabel";
import Checkbox from "@mui/material/Checkbox";
import RadioGroup from "@mui/material/RadioGroup";
import Swal from "sweetalert2";
import Alert from "@mui/material/Alert";

import api from "../utils/api";
import routes from "../utils/routes";
import { apiError } from "../utils/error";
import styles from "../styles/setting.module.scss";
import InputAdornment from "@mui/material/InputAdornment";
import IconButton from "@mui/material/IconButton";
import Visibility from "@mui/icons-material/Visibility";
import VisibilityOff from "@mui/icons-material/VisibilityOff";

import timezones from "../utils/timezones";
import {
  checkNullByteInjection,
  validateName,
  validatePassword,
  validateEmail,
  isEmptyField,
} from "../utils/common";
import { useFeatures } from "../utils/features";
import ConnectTaModal from "../components/connect-ta-modal/connect-ta";
import NewconnecttaModal from "../components/connect-ta-modal/new-connect";

function TabPanel(props) {
  const { children, value, index, ...other } = props;

  return (
    <div
      role="tabpanel"
      hidden={value !== index}
      id={`vertical-tabpanel-${index}`}
      aria-labelledby={`vertical-tab-${index}`}
      {...other}
    >
      {value === index && (
        <Box sx={{ p: 3 }}>
          <Typography>{children}</Typography>
        </Box>
      )}
    </div>
  );
}

TabPanel.propTypes = {
  children: PropTypes.node,
  index: PropTypes.number.isRequired,
  value: PropTypes.number.isRequired,
};

function a11yProps(index) {
  return {
    id: `vertical-tab-${index}`,
    "aria-controls": `vertical-tabpanel-${index}`,
  };
}

export default function Settings(props) {
  const router = useRouter();
  const { query } = router;

  const { canAccessFeature, loadFeatures, areFeaturesLoaded } = useFeatures();
  const [loading, setLoading] = React.useState({
    info: false,
    soc: false,
    password: false,
    apiKey: false,
    testMail: false,
  });
  const [showPassword, setShowPassword] = useState(false);

  const [valueInside, setValueInside] = React.useState(0);

  const [timeZone, setTimeZone] = React.useState("Asia/Kolkata");

  const [taStatus, setTaStatus] = React.useState({
    enabled: false,
    connecting: false,
    isNewCompany: false,
  });

  const [value, setValue] = useState(0);
  const [userInfo, setUserInfo] = React.useState({
    firstName: "",
    lastName: "",
    email: "",
    logo: "",
    companyName: "",
    currentPassword: "",
    newPassword: "",
    confirmPassword: "",
    soc: [],
    apiKey: "",
    licenseKey: "",
    tsat: "",
    isTsat: "",
    smtpFrom: null,
    smtpHost: null,
    smtpUsername: null,
    smtpPassword: null,
    smtpPort: null,
    testMail: null,
    broadcastEnabled: false,
    broadcastThreshold: 0,
    reportingMethod: "Default",
    webhookEmail: "",
    clientId: "",
    secretKey: "",
    tenantId: "",
    senderEmail:"",
    oauthDirectEmailForm: "",
  });
  const [domains, setDomains] = React.useState([]);
  const [webhookEmailDomainId, setWebhookEmailDomainId] = React.useState("");
  const [currentSmtpValue, setCurrentSmtpValue] = React.useState("Default");
  const [smtpId, setSmtpId] = React.useState({
    id: "",
    current: "",
    showClear: false,
  });
  const [getAuthType, setAuthType] = React.useState({
    mail: false,
    totp: false,
    enableAuthenticationType: false,
    is2fa: false,
  });

  const enable2FA = (type) => {
    if (type === "MAIL") {
      setAuthType({ ...getAuthType, mail: true, totp: false });
      if (getAuthType.mail != true) getAuthType.enableAuthenticationType = true;
    } else if (type === "TOTP") {
      setAuthType({ ...getAuthType, totp: true, mail: false });
      if (getAuthType.totp != true) getAuthType.enableAuthenticationType = true;
    }
  };
  const [selectedmicroSoftValue, setSelectedmicroSoftValue] =
    useState("Direct");
  const [taConnectopen, setTaOpen] = React.useState(false);
  const taConnecthandleOpen = () => setTaOpen(true);
  const taConnecthandleClose = () => setTaOpen(false);

  const [openConfigure, setConfigureOpen] = React.useState(false);
  const handleOpenConfigure = (domainId, webhookEmail) => {
    setConfigureOpen(true);
    setWebhookEmailDomainId(domainId);
    setUserInfo({ ...userInfo, webhookEmail });
  };
  const handleConfigureClose = () => {
    setConfigureOpen(false);
    setWebhookEmailDomainId("");
    setUserInfo({ ...userInfo, webhookEmail: "" });
  };

  const [userInfoError, setUserInfoError] = React.useState({
    firstName: { status: false, message: "" },
    lastName: { status: false, message: "" },
    image: { status: false, message: "" },
    companyName: { status: false, message: "" },
    currentPassword: { status: false, message: "" },
    newPassword: { status: false, message: "" },
    confirmPassword: { status: false, message: "" },
    soc: [],
    apiKey: { status: false, message: "" },
    licenseKey: { status: false, message: "" },
    tsat: { status: false, message: "" },
    smtpFrom: { status: false, message: "" },
    smtpHost: { status: false, message: "" },
    smtpUsername: { status: false, message: "" },
    smtpPassword: { status: false, message: "" },
    smtpPort: { status: false, message: "" },
    testMail: { status: false, message: "" },
    broadcastThreshold: { status: false, message: "" },
    webhookEmail: { status: false, message: "" },
    tenantId: { status: false, message: "" },
    secretKey: { status: false, message: "" },
    clientId: { status: false, message: "" },
    oauthDirectEmailForm: { status: false, message: "" },
    senderEmail: { status: false, message: "" },
  });

  const changeErrorState = (field, status, message) => {
    setUserInfoError((prevState) => ({
      ...prevState,
      [field]: { status: status, message: message },
    }));
  };
  const handleTogglePassword = (field) => {
    setShowPassword((prevShowPassword) => ({
      ...prevShowPassword,
      [field]: !prevShowPassword[field],
    }));
  };

  const handleChangeImage = (e) => {
    const file = e.target.files[0];
    if (
      file &&
      (file.type == "image/png" || file.type == "image/jpeg") &&
      !checkNullByteInjection(file.name) &&
      file.size <= 5000000
    )
      setUserInfo({ ...userInfo, logo: e.target.files[0] });
    else {
      setUserInfo({ ...userInfo, logo: "" });
      e.target.value = null;
      toast.error("Only png/jpg image is allowed having size less than 5mb!");
    }
  };

  const handleChange = (event, newValue) => {
    setValue(newValue);
  };

  const handleChangeSoc = (e, indexToChange) => {
    setUserInfo((prevState) => ({
      ...prevState,
      soc: userInfo.soc.map((email, index) => {
        if (index === indexToChange) return e.target.value;
        else return email;
      }),
    }));
    setUserInfoError((prevState) => ({
      ...prevState,
      soc: userInfoError.soc.map(() => ({ status: false, message: "" })),
    }));
  };

  const handleChangeInfo = async (e) => {
    switch (e.target.name) {
      case "firstName": {
        const isValidName = validateName(e.target.value);
        if (!isValidName)
          changeErrorState(
            "firstName",
            true,
            "Name must contain only alphabets and should be 2-30 characters long"
          );
        else changeErrorState("firstName", false, "");
        break;
      }
      case "lastName": {
        const isValidName = validateName(e.target.value);
        if (!isValidName)
          changeErrorState(
            "lastName",
            true,
            "Name must contain only alphabets and should be 2-30 characters long"
          );
        else changeErrorState("lastName", false, "");
        break;
      }
      case "companyName": {
        const CompanyName = /<[^>]*>/.test(e.target.value);
        if (CompanyName)
          changeErrorState(
            "companyName",
            true,
            "Please Enter Valid Company Name"
          );
        else changeErrorState("companyName", false, "");
        break;
      }
      case "currentPassword": {
        const password = e.target.value;
        if (!password || !password.length)
          changeErrorState(
            "currentPassword",
            true,
            "Current Password cannot be empty."
          );
        else changeErrorState("currentPassword", false, "");
        break;
      }
      case "newPassword": {
        const password = e.target.value;
        const isValidPassword = validatePassword(password);
        if (!isValidPassword)
          changeErrorState(
            "newPassword",
            true,
            "Password must contain atleast 8 character, 1 Uppercase, 1 Lowercase, 1 digit, 1 special character & no whitespaces are allowed at beginning & end."
          );
        else changeErrorState("newPassword", false, "");
        if (!(password == userInfo.confirmPassword)) {
          changeErrorState("confirmPassword", true, "Password Does not Match!");
        } else changeErrorState("confirmPassword", false, "");
        break;
      }
      case "confirmPassword": {
        const password = e.target.value;
        if (!(password == userInfo.newPassword))
          changeErrorState("confirmPassword", true, "Password Does not Match!");
        else changeErrorState("confirmPassword", false, "");
        break;
      }
      // case 'soc' : {
      // 	const email = e.target.value;
      // 	if(!validateEmail(email)) changeErrorState('soc',true,'Please Enter Valid Email')
      // 	else changeErrorState('soc',false,'')
      // 	break;
      // }
      case "webhookEmail": {
        const email = e.target.value;
        if (!validateEmail(email))
          changeErrorState(
            "webhookEmail",
            true,
            "Please Enter a valid email address!"
          );
        else changeErrorState("webhookEmail", false, "");
        break;
      }
      case "apiKey": {
        const apiKey = e.target.value;
        if (!isEmptyField(apiKey))
          changeErrorState("apiKey", true, "Api Key cannot be empty");
        else changeErrorState("apiKey", false, "");
        break;
      }
      case "broadcastThreshold": {
        const threshold = e.target.value;

        if (threshold == 0)
          changeErrorState(
            "broadcastThreshold",
            true,
            "Please select a threshold value greater than 0."
          );
        else changeErrorState("broadcastThreshold", false, "");

        break;
      }
      case "licenseKey": {
        const licenseKey = e.target.value;

        if (!isEmptyField(licenseKey))
          changeErrorState("licenseKey", true, "License cannot be empty");
        else changeErrorState("licenseKey", false, "");

        break;
      }
      case "tsat": {
        const tsat = e.target.value;

        if (!isEmptyField(tsat))
          changeErrorState("tsat", true, "URL cannot be empty!");
        else changeErrorState("tsat", false, "");

        break;
      }
      case "smtpName": {
        const smtpName = e.target.value;

        if (!isEmptyField(smtpName))
          changeErrorState("smtpName", true, "Name cannot be empty!");
        else changeErrorState("smtpName", false, "");

        break;
      }
      case "smtpFrom": {
        const smtpFrom = e.target.value;

        if (!isEmptyField(smtpFrom))
          changeErrorState("smtpFrom", true, "From cannot be empty!");
        else changeErrorState("smtpFrom", false, "");

        break;
      }
      case "smtpHost": {
        const smtpHost = e.target.value;

        if (!isEmptyField(smtpHost))
          changeErrorState("smtpHost", true, "Host cannot be empty!");
        else changeErrorState("smtpHost", false, "");

        break;
      }
      case "smtpUsername": {
        const smtpUsername = e.target.value;

        if (!isEmptyField(smtpUsername))
          changeErrorState("smtpUsername", true, "Username cannot be empty!");
        else changeErrorState("smtpUsername", false, "");

        break;
      }
      case "smtpPassword": {
        const smtpPassword = e.target.value;

        if (!isEmptyField(smtpPassword))
          changeErrorState("smtpPassword", true, "Password cannot be empty!");
        else changeErrorState("smtpPassword", false, "");

        break;
      }
      case "smtpPort": {
        const smtpPort = e.target.value;

        if (!isEmptyField(smtpPort))
          changeErrorState("smtpPort", true, "Port cannot be empty!");
        else changeErrorState("smtpPort", false, "");

        break;
      }
      case "testMail": {
        const smtpPort = e.target.value;

        if (!isEmptyField(smtpPort))
          changeErrorState("testMail", true, "Email Cannot be Empty!");
        else changeErrorState("testMail", false, "");

        break;
      }

      case "clientId": {
        const clientId = e.target.value;
        if (!isEmptyField(clientId)) {
          changeErrorState("clientId", true, "Client Id Cannot be Empty!");
        } else changeErrorState("clientId", false, "");
      }

      case "secretKey": {
        const secretKey = e.target.value;
        if (!isEmptyField(secretKey)) {
          changeErrorState("secretKey", true, "Secret Key Cannot be Empty!");
        } else changeErrorState("secretKey", false, "");
      }

      case "tenantId": {
        const tenantId = e.target.value;
        if (!isEmptyField(tenantId)) {
          changeErrorState("tenantId", true, "Tenant Id Cannot be Empty!");
        } else changeErrorState("tenantId", false, "");
      }

      case "senderEmail": {
        const senderEmail = e.target.value;
        if (!validateEmail(senderEmail)) {
          changeErrorState(
            "senderEmail",
            true,
            "Please Enter a valid email address!"
          );
        } else {
          changeErrorState("senderEmail", false, "");
        }
        break;
      }

      case "oauthDirectEmailForm": {
        const email = e.target.value;
        if (!validateEmail(email)) {
          changeErrorState(
            "oauthDirectEmailForm",
            true,
            "Please Enter a valid email address!"
          );
        } else {
          changeErrorState("oauthDirectEmailForm", false, "");
        }
        break;
      }
    }
    setUserInfo({ ...userInfo, [e.target.name]: e.target.value });
  };

  const handleClick = async (e) => {
    switch (e.target.id) {
      case "user": {
        const { firstName, lastName, companyName, logo } = userInfo;

        if (
          validateName(firstName) &&
          validateName(lastName) &&
          !/<[^>]*>/.test(companyName)
        ) {
          const formData = new FormData();
          formData.append("firstName", firstName);
          formData.append("lastName", lastName);
          formData.append("companyName", companyName);
          if (logo !== "") formData.append("logo", logo);

          if (firstName.length && lastName.length && companyName.length) {
            setLoading((prevState) => ({ ...prevState, info: true }));
            api
              .post(routes.SETTINGS.UPDATE_PROFILE, formData, {
                headers: {
                  "Content-Type": "multipart/form-data",
                },
              })
              .then((res) => res.data)
              .then((data) => {
                const expires = new Date(
                  Cookies.get("username", { raw: true }).expires
                );
                Cookies.set("username", firstName, { expires });
                Cookies.set("logo", data.dataUrl, { expires });
                setLoading((prevState) => ({ ...prevState, info: false }));
                toast.success(data.message);
              })
              .catch((error) => {
                setLoading((prevState) => ({ ...prevState, info: false }));
                apiError(error);
              });
          }
        }
        break;
      }

      case "password": {
        const { currentPassword, newPassword, confirmPassword } = userInfo;
        const error = false;
        if (!currentPassword || !currentPassword.length) {
          error = true;
          changeErrorState(
            "currentPassword",
            true,
            "Password cannot be empty!"
          );
        }
        if (
          !newPassword ||
          !newPassword.length ||
          !validatePassword(newPassword)
        ) {
          error = true;
          changeErrorState(
            "newPassword",
            true,
            "Password must contain atleast 8 character, 1 Uppercase, 1 Lowercase, 1 digit, 1 special character & no whitespaces are allowed at beginning & end."
          );
        }
        if (
          !confirmPassword ||
          !confirmPassword.length ||
          newPassword !== confirmPassword
        ) {
          error = true;
          changeErrorState(
            "confirmPassword",
            true,
            "Passwords does not match."
          );
        }
        if (
          !confirmPassword ||
          !confirmPassword.length ||
          newPassword == currentPassword
        ) {
          error = true;
          changeErrorState(
            "newPassword",
            true,
            "New Password must differ from current password!"
          );
        }
        if (!error) {
          setLoading((prevState) => ({ ...prevState, password: true }));
          api
            .post(routes.SETTINGS.CHANGE_PASSWORD, {
              currentpassword: userInfo.currentPassword,
              password: userInfo.newPassword,
            })
            .then((res) => res.data)
            .then((data) => {
              setLoading((prevState) => ({ ...prevState, password: false }));
              toast.success(data.Message);
              setUserInfo({
                ...userInfo,
                currentPassword: "",
                newPassword: "",
                confirmPassword: "",
              });
              setTimeout(() => {
                router.push("/login");
              }, 1000);
            })
            .catch((error) => {
              setLoading((prevState) => ({ ...prevState, password: false }));
              apiError(error);
            });
        }
        break;
      }
      case "socMail": {
        const { soc } = userInfo;
        const error = false;

        for (let i = 0; i < userInfo.soc.length; i++) {
          if (!validateEmail(userInfo.soc[i])) {
            error = true;
            setUserInfoError((prevState) => ({
              ...prevState,
              soc: userInfoError.soc.map((element, index) =>
                index === i
                  ? { status: true, message: "Please Enter a valid mail" }
                  : element
              ),
            }));
          }
        }
        if (!error) {
          setLoading((prevState) => ({ ...prevState, soc: "true" }));
          api
            .patch(routes.SETTINGS.UPDATE_SOC, { socMail: soc })
            .then((res) => res.data)
            .then((data) => {
              setLoading((prevState) => ({ ...prevState, soc: false }));
              toast.success("SOC-Mail Updated Successfully");
            })
            .catch((error) => {
              setLoading((prevState) => ({ ...prevState, soc: false }));
              apiError(error);
            });
        }
        break;
      }
      case "webhookEmail": {
        const { webhookEmail } = userInfo;
        const error = false;
        if (!validateEmail(webhookEmail)) {
          error = true;
          changeErrorState(
            "webhookEmail",
            true,
            "Please enter a valid email address!"
          );
        }
        if (!error) {
          api
            .post(routes.DOMAIN.UPDATE_WEBHOOK_EMAIL_FOR_DOMAIN, {
              email: webhookEmail,
              domainId: webhookEmailDomainId,
            })
            .then((res) => res.data)
            .then((data) => {
              toast.success(data.message);
              setUserInfo({ ...userInfo, webhookEmail: "" });
              setWebhookEmailDomainId("");
              setConfigureOpen(false);
            })
            .catch((error) => {
              apiError(error);
            });
        }
        break;
      }
      case "apiKey": {
        const { apiKey } = userInfo;
        const error = false;

        if (!isEmptyField(apiKey)) {
          error = true;
          changeErrorState("apiKey", true, "Api Key cannot be empty");
        }
        if (!error) {
          setLoading((prevState) => ({ ...prevState, apiKey: true }));
          api
            .post(routes.SETTINGS.API_KEY, { apikey: apiKey })
            .then((res) => res.data)
            .then((data) => {
              setLoading((prevState) => ({ ...prevState, apiKey: false }));
              toast.success("Api-Key Updated Successfully");
              setUserInfo((prevState) => ({ ...prevState, apiKey: "" }));
            })
            .catch((error) => {
              setLoading((prevState) => ({ ...prevState, apiKey: false }));
              apiError(error);
            });
        }
        break;
      }
      case "broadcastThreshold": {
        const { broadcastEnabled, broadcastThreshold } = userInfo;
        const error = false;

        if (broadcastEnabled && broadcastThreshold == 0) {
          error = true;
          changeErrorState(
            "broadcastThreshold",
            true,
            "Please select a threshold value greater than 0."
          );
        }
        if (!error) {
          api
            .post(routes.SETTINGS.UPDATE_BROADCAST_STATUS, {
              broadcastEnabled,
              broadcastThreshold,
            })
            .then((res) => res.data)
            .then((data) => {
              toast.success(data.message);
            })
            .catch((error) => {
              apiError(error);
            });
        }
        break;
      }
      case "licenseKey": {
        const { licenseKey } = userInfo;
        const error = false;

        if (!isEmptyField(licenseKey)) {
          error = true;
          changeErrorState("licenseKey", true, "License Key cannot be empty");
        }
        if (!error) {
          setLoading((prevState) => ({ ...prevState, licenseKey: true }));
          api
            .post(routes.SETTINGS.LICENSE_KEY, { licenceKey: licenseKey })
            .then((res) => res.data)
            .then((data) => {
              setLoading((prevState) => ({ ...prevState, licenseKey: false }));
              toast.success("License-Key Updated Successfully");
            })
            .catch((error) => {
              setLoading((prevState) => ({ ...prevState, licenseKey: false }));
              apiError(error);
            });
        }
        break;
      }
      case "tsat": {
        const { tsat } = userInfo;
        const error = false;

        if (!isEmptyField(tsat)) {
          error = true;
          changeErrorState("tsat", true, "URL cannot be empty!");
        }
        if (!error) {
          api
            .post(routes.SETTINGS.TSAT_URL, { tsatUrl: userInfo.tsat })
            .then((res) => res.data)
            .then((data) => {
              // setUserInfo((prevState) => ({...prevState, tsat: data.tsatUrl }))
              loadDetails();
              toast.success(data.message);
            })
            .catch((error) => {
              apiError(error);
            });
        }
        break;
      }

      case "defaultSmtp": {
        api
          .post(routes.SETTINGS.UPDATE_SMTP, { mailSmtp: false })
          .then((res) => res.data)
          .then((data) => {
            loadDetails();
            toast.success(data.message);
          })
          .catch((error) => {
            apiError(error);
          });
        break;
      }
      case "customSmtp": {
        const { smtpFrom, smtpHost, smtpUsername, smtpPassword, smtpPort } =
          userInfo;
        let error = false;

        if (!smtpFrom) {
          error = true;
          changeErrorState("smtpFrom", true, "From cannot be empty");
        }
        if (!smtpHost) {
          error = true;
          changeErrorState("smtpHost", true, "Host cannot be empty");
        }
        if (!smtpUsername) {
          error = true;
          changeErrorState("smtpUsername", true, "Username cannot be empty");
        }
        if (!smtpPassword) {
          error = true;
          changeErrorState("smtpPassword", true, "Password cannot be empty");
        }
        if (!smtpPort) {
          error = true;
          changeErrorState("smtpPort", true, "Port cannot be empty");
        }
        if (!error) {
          api
            .post(routes.SETTINGS.UPDATE_SMTP, {
              smtpFrom,
              smtpHost,
              smtpUsername,
              smtpPassword,
              smtpPort,
              mailSmtp: true,
            })
            .then((res) => res.data)
            .then((data) => {
              loadDetails();
              toast.success(data.message);
              smpthandleClose();
            })
            .catch((error) => {
              apiError(error);
            });
        }
        break;
      }
      case "smtpModal": {
        const { smtpFrom, smtpHost, smtpUsername, smtpPassword, smtpPort } =
          userInfo;
        let error = false;

        if (!smtpFrom) {
          error = true;
          changeErrorState("smtpFrom", true, "From cannot be empty");
        }
        if (!smtpHost) {
          error = true;
          changeErrorState("smtpHost", true, "Host cannot be empty");
        }
        if (!smtpUsername) {
          error = true;
          changeErrorState("smtpUsername", true, "Username cannot be empty");
        }
        if (!smtpPassword) {
          error = true;
          changeErrorState("smtpPassword", true, "Password cannot be empty");
        }
        if (!smtpPort) {
          error = true;
          changeErrorState("smtpPort", true, "Port cannot be empty");
        }

        if (!error) smtphandleOpen();
        break;
      }
      case "sendTestMail": {
        const {
          // smtpFrom,
          // smtpHost,
          // smtpUsername,
          // smtpPassword,
          // smtpPort,
          testMail,
        } = userInfo;
        let error = false;

        if (!testMail) {
          error = true;
          changeErrorState("testMail", true, "Email Id Cannot be Empty!");
        }
        if (!error) {
          setLoading((prevState) => ({ ...prevState, testMail: true }));
          api
            .post(routes.SETTINGS.SEND_TEST_MAIL, {
              // smtpFrom,
              // smtpHost,
              // smtpUsername,
              // smtpPassword,
              // smtpPort,
              testMail,
            })
            .then((res) => res.data)
            .then((data) => {
              setLoading((prevState) => ({ ...prevState, testMail: false }));
              toast.success(data.message);
              smpthandleClose();
            })
            .catch((error) => {
              setLoading((prevState) => ({ ...prevState, testMail: false }));
              apiError(error);
            });
        }
        break;
      }
      case "directOauth": {
        const { oauthDirectEmailForm } = userInfo;
        let error = false;
        if (!validateEmail(oauthDirectEmailForm)) {
          error = true;
          changeErrorState(
            "oauthDirectEmailForm",
            true,
            "Please Enter a valid email address!"
          );
        }
        if (!error) {
          await api
            .post(routes.SETTINGS.OAUTH_DIRECT_EMAIL, {
              senderEmail: userInfo.oauthDirectEmailForm,
            })
            .then((res) => res.data)
            .then((data) => {
              toast.success(data.message);
              setTimeout(() => {
                window.location.href = data.data.consentUrl;
              }, 2000);
            })
            .catch((error) => {
              apiError(error);
            });
        }
        break;
      }

      case "oauthManualSave": {
        const { clientId, secretKey, tenantId } = userInfo;
        let error = false;
        if (!isEmptyField(clientId.trim())) {
          error = true;
          changeErrorState("clientId", true, "Client Id Cannot be Empty!");
        }
        if (!isEmptyField(secretKey.trim())) {
          error = true;
          changeErrorState("secretKey", true, "Client Id Cannot be Empty!");
        }
        if (!isEmptyField(tenantId.trim())) {
          error = true;
          changeErrorState("tenantId", true, "Tenant Id Cannot be Empty!");
        }
        if (!error) {
          await api
            .post(routes.SETTINGS.UPDATE_SMTP_OAUTH_MANUAL, {
              clientId: userInfo.clientId,
              clientSecret: userInfo.secretKey,
              tenantId: userInfo.tenantId,
              senderEmail: userInfo.senderEmail,
            })
            .then((res) => {
              if (res.status == 200) {
                const resposne = res.data;
                toast.success(resposne.message);
                getSmtpDetail();
              }
            })
            .catch((err) => apiError(err));
        }
        break;
      }

      case "clearOauth": {
        Swal.fire({
          title: "Are you sure?",
          text: "This will clear the OAuth configuration and you will need to reconfigure it again.",
          icon: "warning",
          showCancelButton: true,
          confirmButtonText: "Yes, clear it!",
          cancelButtonText: "No, keep it",
          reverseButtons: true,
        }).then(async (result) => {
          if (result.isConfirmed) {
            await api
              .delete(routes.SETTINGS.CLEAR_OAUTH(smtpId.id))
              .then((res) => res.data)
              .then((data) => {
                toast.success(data.message);
                setSmtpId({ id: "", current: "", showClear: false });
                setUserInfo((prevState) => ({
                  ...prevState,
                  oauthDirectEmailForm: "",
                  clientId: "",
                  secretKey: "",
                  tenantId: "",
                  senderEmail: "",
                }));
              })
              .catch((error) => {
                apiError(error);
            });
          }
        });
        break;
      }

      case "clearCustomSmtp":{
        Swal.fire({
          title: "Are you sure?",
          text: "This will clear the Custome SMTP.",
          icon: "warning",
          showCancelButton: true,
          confirmButtonText: "Yes, clear it!",
          cancelButtonText: "No, keep it",
          reverseButtons: true,
        }).then(async (result) => {
          if(result.isConfirmed) {
            await api.post(routes.SETTINGS.DELETE_CUSTOM_SMTP)
            .then((res) => res.data)
            .then(response=>{
              toast.success(response.message);
              setUserInfo((prevState) => ({
                ...prevState,
                smtpFrom: "",
                smtpHost: "",
                smtpUsername: "",
                smtpPassword: "",
                smtpPort: "",
              }));
            })
            .catch(err=>apiError(err));
            
          }
          
        })
        break;
      }
    }
  };

  const changeStatus = (status) => {
    api
      .post(routes.SETTINGS.TSAT_URL, { status: !status })
      .then((res) => res.data)
      .then((data) => {
        setUserInfo((prevState) => ({ ...prevState, isTsat: data.status }));
        toast.success(data.message);
      })
      .catch((error) => {
        apiError(error);
      });
  };

  const changeBroadcastStatus = (status) => {
    if (status) {
      api
        .post(routes.SETTINGS.UPDATE_BROADCAST_STATUS, {
          broadcastEnabled: false,
          broadcastThreshold: 0,
        })
        .then((res) => res.data)
        .then((data) => {
          setUserInfo((prevState) => ({
            ...prevState,
            broadcastThreshold: 0,
            broadcastEnabled: false,
          }));
          toast.success(data.message);
        })
        .catch((error) => {
          apiError(error);
        });
    }
    setUserInfo((prevState) => ({ ...prevState, broadcastEnabled: !status }));
  };

  const [disableState, setDisableState] = React.useState("");
  const loadDetails = () => {
    api
      .get(routes.SETTINGS.PROFILE_DETAILS)
      .then((res) => res.data)
      .then((data) => {
        setUserInfo((prevState) => ({
          ...prevState,
          firstName: data.user.firstName,
        }));
        setUserInfo((prevState) => ({
          ...prevState,
          lastName: data.user.lastName,
        }));
        setUserInfo((prevState) => ({ ...prevState, email: data.user.email }));
        setUserInfo((prevState) => ({
          ...prevState,
          companyName: data.company.name,
        }));
        setUserInfo((prevState) => ({
          ...prevState,
          soc: data.company.socEmail,
        }));
        setUserInfoError((prevState) => ({
          ...prevState,
          soc: data.company.socEmail.map(() => {
            return { status: false, message: "" };
          }),
        }));
        setUserInfo((prevState) => ({
          ...prevState,
          licenseKey: data.company.licenceKey,
        }));
        setUserInfo((prevState) => ({ ...prevState, tsat: data.tsatUrl }));
        setUserInfo((prevState) => ({
          ...prevState,
          isTsat: JSON.parse(data.isTsat),
        }));
        setUserInfo((prevState) => ({
          ...prevState,
          broadcastThreshold: data.company.broadcastThreshold,
        }));
        setUserInfo((prevState) => ({
          ...prevState,
          broadcastEnabled: data.company.broadcastThreshold == 0 ? false : true,
        }));
        setUserInfo((prevState) => ({
          ...prevState,
          reportingMethod: data.company.reportingMethod,
        }));
        setTimeZone(data.timeZone);

        if (data.user.is2fa) {
          if (data.user.otpType == "MAIL") {
            setAuthType((prevState) => ({
              ...prevState,
              mail: true,
              totp: false,
              is2fa: data.user.is2fa,
            }));
            setDisableState("mail");
          } else {
            setAuthType((prevState) => ({
              ...prevState,
              mail: false,
              totp: true,
              is2fa: data.user.is2fa,
            }));
            setDisableState("totp");
          }
        }
        if (data.company.logo) {
          let expiredTime = new Date(
            new Date().setMinutes(new Date().getMinutes() + 180)
          );
          Cookies.set("logo", `${data.url}${data.company.logo}`, {
            expires: expiredTime,
            httpOnly: true,
          });
        }
        if (data.smtp.status) {
          setsmptHideShiw(false);
          setcustomsmsptHideShiw(true);
          setUserInfo((prevState) => ({
            ...prevState,
            smtpName: data.smtp.name,
          }));
          setUserInfo((prevState) => ({
            ...prevState,
            smtpFrom: data.smtp.from,
          }));
          setUserInfo((prevState) => ({
            ...prevState,
            smtpHost: data.smtp.host,
          }));
          setUserInfo((prevState) => ({
            ...prevState,
            smtpUsername: data.smtp.username,
          }));
          setUserInfo((prevState) => ({ ...prevState, smtpPassword: "****" }));
          setUserInfo((prevState) => ({
            ...prevState,
            smtpPort: data.smtp.port,
          }));
        } else {
          setsmptHideShiw(true);
          setcustomsmsptHideShiw(false);
        }
      })
      .catch((error) => {
        apiError(error);
      });
  };

  const [templates, setTemplates] = React.useState([]);

  const loadTemplates = () => {
    api
      .get(routes.SETTINGS.LOAD_TEMPLATES)
      .then((res) => res.data)
      .then((data) => {
        setTemplates(data.templates);
      });
  };

  const checkTaConnection = () => {
    api
      .get(routes.TA.CHECK_CONNECTION)
      .then((res) => res.data)
      .then((data) => {
        setTaStatus({ ...taStatus, ...data.taStatus });
      })
      .catch((error) => {
        console.log("Error from TA-Connection", error);
      });
  };

  const getSmtpDetail = () => {
    api
      .get(routes.SETTINGS.GET_SMTP_DETAILS)
      .then((res) => res.data)
      .then((response) => {
        if (response.data.oauthConfig) {
          const { senderEmail, id } = response.data.oauthConfig;
          setSmtpId({ id, current: "direct", showClear: true });
          setSelectedmicroSoftValue("Direct");
          setUserInfo((prevState) => ({
            ...prevState,
            oauthDirectEmailForm: senderEmail,
          }));
          setCurrentSmtpValue("Oauth");
          handleCheckboxChangecustomOauth();
        }
        if (response.data.manualConfig) {
          const { tenantId, clientId, secretId, id,senderEmail } =
            response.data.manualConfig;
          setSmtpId({ id, current: "manual", showClear: true });
          setSelectedmicroSoftValue("Manual");
          setUserInfo((prevState) => ({
            ...prevState,
            tenantId: tenantId,
            clientId: clientId,
            secretKey: secretId,
            senderEmail: senderEmail,
          }));
          setCurrentSmtpValue("Oauth");
          handleCheckboxChangecustomOauth();
        }
        
      })
      .catch((error) => {
        apiError(error);
      });
  };

  React.useEffect(() => {
    checkTaConnection();
  }, []);

  React.useEffect(() => {
    loadFeatures();
    getSmtpDetail();
  }, []);

  React.useEffect(() => {
    loadDetails();
  }, []);
  React.useEffect(() => {
    loadTemplates();
  }, []);

  React.useEffect(() => {
    if (userInfo.reportingMethod == "Email Client") {
      api
        .get(routes.DOMAIN.LIST)
        .then((res) => res.data)
        .then((data) => {
          let domains = data.domains.filter(
            (domain) => domain.domainTypeForOutlook == "OFFICE365"
          );
          setDomains(domains);
        });
    }
  }, [userInfo.reportingMethod]);

  React.useEffect(() => {
    if ("smtp" in query) {
      setValue(taStatus.enabled ? 4 : 6);
      setCurrentSmtpValue("Oauth");
      handleCheckboxChangecustomOauth();
      if(router.query.success){
        toast.success(router.query.message);
      }
    }
  }, [query]);

  const disable2FA = () => {
    api
      .post(routes.TWOFA.DISABLED)
      .then((res) => res.data)
      .then((data) => {
        toast.success(data.message);
        setAuthType((prevState) => ({
          ...prevState,
          mail: false,
          totp: false,
          is2fa: false,
        }));
        handleClose();
      })
      .catch((error) => {
        apiError(error);
      });
  };

  const set2FA = () => {
    if (getAuthType.mail || getAuthType.totp) {
      if (getAuthType.mail) {
        api
          .post(routes.TWOFA.MAIL)
          .then((res) => res.data)
          .then((data) => {
            toast.success(data.message);
            loadDetails();
            handleClose();
          })
          .catch((error) => {
            apiError(error);
          });
      } else {
        router.push("/auth");
      }
    } else {
      toast.error("Please select 2FA type");
    }
  };

  const updateTimezone = () => {
    api
      .post(routes.SETTINGS.UPDATE_TIMEZONE, { timeZone })
      .then((res) => res.data)
      .then((data) => {
        toast.success(data.message);
      });
  };

  const disconnectSwal = () => {
    Swal.fire({
      icon: "info",
      title: "Disconnect TPIR",
      text: "To disconnect TPIR from Threatcop Admin, navigate to the Application Management section in Threatcop Admin.",
      confirmButtonText: "Close",
      confirmButtonColor: "#6c757d",
    });
  };

  const [openModal, setOpen] = React.useState(false);
  const handleOpen = () => setOpen(true);
  const handleClose = () => setOpen(false);

  const [smptopenModal, setsmptOpen] = React.useState(false);
  const smtphandleOpen = () => setsmptOpen(true);
  const smpthandleClose = () => {
    setsmptOpen(false);
    setUserInfoError((prevState) => ({
      ...prevState,
      testMail: { status: false, message: "" },
    }));
  };

  const [defultsmptHideShiw, setsmptHideShiw] = React.useState(true);
  const [customsmptHideShiw, setcustomsmsptHideShiw] = React.useState(false);
  const [showCustomOauth, setShowCustomOauth] = React.useState(false);

  const handleCheckboxChange = () => {
    setsmptHideShiw(true);
    setcustomsmsptHideShiw(false);
    setShowCustomOauth(false);
  };

  const handleCheckboxChangecustom = () => {
    setcustomsmsptHideShiw(true);
    setsmptHideShiw(false);
    setShowCustomOauth(false);
  };

  const handleCheckboxChangecustomOauth = () => {
    setcustomsmsptHideShiw(false);
    setsmptHideShiw(false);
    setShowCustomOauth(true);
  };

  const handleChangeSmtp = (e) => {
    const value = e.target.value;
    setCurrentSmtpValue(value);
  };

  const handleRemoveSoc = (indexToRemove) => {
    setUserInfo((prevState) => ({
      ...prevState,
      soc: userInfo.soc.filter((email, index) => index !== indexToRemove),
    }));
    setUserInfoError((prevState) => ({
      ...prevState,
      soc: userInfoError.soc.filter((email, index) => index !== indexToRemove),
    }));
  };

  const handleAddSoc = (indexToRemove) => {
    let newSoc = userInfo.soc;
    newSoc.push("");
    let newSocError = userInfoError.soc;
    newSocError.push({ status: false, message: "" });

    setUserInfo((prevState) => ({ ...prevState, soc: newSoc }));
    setUserInfoError((prevState) => ({ ...prevState, soc: newSocError }));
  };

  const [loadingTaConnection, setLoadingTaConnection] = React.useState(false);

  const changeConnectionStatusWithTa = async (status) => {
    setLoadingTaConnection(true);
    try {
      setTaStatus({ ...taStatus, enabled: false, connecting: status });

      const response = await api.post(routes.TA.CONNECT, {
        connectionStatus: status,
      });
      const { success } = response.data;

      if (success) {
        toast.success(response.data.message);
        if (status) setTaOpen(false);
      }
    } catch (error) {
      console.error("Error from TA-Connection:", error);
    } finally {
      setLoadingTaConnection(false);
    }
  };

  const updateReportingMethod = (method) => {
    if (method == "default")
      setUserInfo({ ...userInfo, reportingMethod: "Default" });
    else setUserInfo({ ...userInfo, reportingMethod: "Email Client" });
    api
      .post(routes.SETTINGS.UPDATE_REPORTING_METHOD, {
        defaultReportingMethod: method == "default" ? true : false,
      })
      .then((res) => res.data)
      .then((data) => {
        if (method == "default")
          setUserInfo({ ...userInfo, reportingMethod: "Default" });
        else setUserInfo({ ...userInfo, reportingMethod: "Email Client" });
        toast.success(data.message);
      })
      .catch((error) => {
        apiError(error);
      });
  };

  const handleChangemicroSoft = (event) => {
    setSelectedmicroSoftValue(event.target.value);
  };

  return !areFeaturesLoaded ? (
    <box>Loading</box>
  ) : (
    <Box className={`${styles.settingWrap} ${styles.containerbar}`}>
      <Box className={styles.containerbar}>
        <Box className={styles.breadcrumbBar}>
          <Grid
            container
            direction="row"
            justifyContent="center"
            alignItems="center"
          >
            <Grid item xs={4}>
              <Typography
                component="h4"
                variant="h4"
                className={styles.page_title}
              >
                Settings
              </Typography>
            </Grid>
            <Grid item xs={8}>
              {/* <Box className={styles.flex_btn}>
                  <Button variant="contained" className={styles.btn_new_project} onClick={handleOpen}> Two Factor Authentication </Button>
                </Box> */}
            </Grid>
          </Grid>
        </Box>

        <Box className={styles.settingTabBar}>
          <Box sx={{ width: "100%" }}>
            <Box className={styles.tabItem}>
              <Tabs
                value={value}
                onChange={handleChange}
                aria-label="basic tabs example"
                className={styles.TabItemButton}
              >
                {!taStatus.enabled && <Tab label="Profile" {...a11yProps(0)} />}
                {!canAccessFeature("TPIR-EASOC") ? (
                  <Tooltip
                    title="Please upgrade to unlock this functionality!"
                    arrow
                  >
                    <span>
                      <Tab
                        label={
                          <Box sx={{ display: "flex", alignItems: "center" }}>
                            SOC Email
                          </Box>
                        }
                        {...a11yProps(1)}
                        disabled
                      />
                    </span>
                  </Tooltip>
                ) : (
                  <Tab
                    label="SOC Email"
                    {...a11yProps(taStatus.enabled ? 0 : 1)}
                  />
                )}
                {!taStatus.enabled && (
                  <Tab label="Password" {...a11yProps(2)} />
                )}
                {!canAccessFeature("TPIR-EB") ? (
                  <Tooltip
                    title="Please upgrade to unlock this functionality"
                    arrow
                  >
                    <span>
                      <Tab
                        label={
                          <Box sx={{ display: "flex", alignItems: "center" }}>
                            Broadcast
                          </Box>
                        }
                        {...a11yProps(3)}
                        disabled
                      />
                    </span>
                  </Tooltip>
                ) : (
                  <Tab
                    label="Broadcast"
                    {...a11yProps(taStatus.enabled ? 1 : 3)}
                  />
                )}
                {!canAccessFeature("TPIR-AALP") ? (
                  <Tooltip
                    title="Please upgrade to unlock this functionality"
                    arrow
                  >
                    <span>
                      <Tab
                        label={
                          <Box sx={{ display: "flex", alignItems: "center" }}>
                            Virus Total
                          </Box>
                        }
                        {...a11yProps(4)}
                        disabled
                      />
                    </span>
                  </Tooltip>
                ) : (
                  <Tab
                    label="Virus Total"
                    {...a11yProps(taStatus.enabled ? 2 : 4)}
                  />
                )}
                {!canAccessFeature("TPIR-TSAT") ? (
                  <Tooltip
                    title="Please upgrade to unlock this functionality"
                    arrow
                  >
                    <span>
                      <Tab
                        label={
                          <Box sx={{ display: "flex", alignItems: "center" }}>
                            TSAT Config
                          </Box>
                        }
                        {...a11yProps(5)}
                        disabled
                      />
                    </span>
                  </Tooltip>
                ) : (
                  <Tab
                    label="TSAT Config"
                    {...a11yProps(taStatus.enabled ? 3 : 5)}
                  />
                )}
                {!canAccessFeature("TPIR-CSTM") ? (
                  <Tooltip
                    title="Please upgrade to unlock this functionality"
                    arrow
                  >
                    <span>
                      <Tab
                        label={
                          <Box sx={{ display: "flex", alignItems: "center" }}>
                            SMTP
                          </Box>
                        }
                        {...a11yProps(6)}
                        disabled
                      />
                    </span>
                  </Tooltip>
                ) : (
                  <Tab label="SMTP" {...a11yProps(taStatus.enabled ? 4 : 6)} />
                )}
                {!canAccessFeature("TPIR-CET") ? (
                  <Tooltip
                    title="Please upgrade to unlock this functionality"
                    arrow
                  >
                    <span>
                      <Tab
                        label={
                          <Box sx={{ display: "flex", alignItems: "center" }}>
                            Templates
                          </Box>
                        }
                        {...a11yProps(7)}
                        disabled
                      />
                    </span>
                  </Tooltip>
                ) : (
                  <Tab
                    label="Templates"
                    {...a11yProps(taStatus.enabled ? 5 : 7)}
                  />
                )}
                <Tab
                  label="Timezone"
                  {...a11yProps(taStatus.enabled ? 6 : 8)}
                />

                {!taStatus.enabled && (
                  <Tab label="Security" {...a11yProps(9)} />
                )}
                {!taStatus.isNewCompany && (
                  <Tab
                    label="Threatcop Admin"
                    {...a11yProps(taStatus.enabled ? 7 : 10)}
                  />
                )}

                <Tab
                  label={
                    <Box sx={{ display: "flex", alignItems: "center" }}>
                      Reporting Method
                    </Box>
                  }
                  {...a11yProps(11)}
                />
              </Tabs>
            </Box>

            {!taStatus.enabled && (
              <TabPanel
                value={value}
                index={0}
                className={styles.commontabPanel}
              >
                <Box className={styles.tabHeading}>
                  <Typography component="p">
                    User-specific settings and personal information stored
                    within the system.
                  </Typography>
                </Box>

                <Grid container className={styles.settingForm}>
                  <Grid item md={4} lg={4}>
                    <InputLabel>First Name</InputLabel>
                    <TextField
                      size="small"
                      type="text"
                      placeholder="Enter First Name"
                      name="firstName"
                      error={userInfoError.firstName.status}
                      helperText={userInfoError.firstName.message}
                      value={userInfo.firstName}
                      onChange={handleChangeInfo}
                      fullWidth
                      sx={{ mb: "20px" }}
                    />

                    <InputLabel>Last Name</InputLabel>
                    <TextField
                      size="small"
                      type="text"
                      placeholder="Enter Last Name"
                      name="lastName"
                      error={userInfoError.lastName.status}
                      helperText={userInfoError.lastName.message}
                      value={userInfo.lastName}
                      onChange={handleChangeInfo}
                      fullWidth
                      sx={{ mb: "20px" }}
                    />

                    <InputLabel>Company Name</InputLabel>
                    <TextField
                      size="small"
                      type="text"
                      placeholder="Enter Company Name"
                      name="companyName"
                      value={userInfo.companyName}
                      error={userInfoError.companyName.status}
                      helperText={userInfoError.companyName.message}
                      onChange={handleChangeInfo}
                      fullWidth
                      sx={{ mb: "20px" }}
                    />

                    <InputLabel>Email</InputLabel>
                    <TextField
                      size="small"
                      placeholder="Email ID "
                      value={userInfo.email}
                      disabled={true}
                      name="email"
                      fullWidth
                      sx={{ mb: "20px" }}
                    />
                    <InputLabel>Upload Image</InputLabel>

                    <TextField
                      size="small"
                      type="file"
                      id="myFile"
                      name="filename"
                      onChange={handleChangeImage}
                      accept="image/*"
                      fullWidth
                      sx={{ mb: "14px" }}
                    />
                    <LoadingButton
                      className={styles.btnSuccess}
                      id={"user"}
                      disabled={loading.info}
                      loading={loading.info}
                      loadingPosition="start"
                      variant="contained"
                      sx={{ mt: "10px" }}
                      onClick={handleClick}
                    >
                      Save Changes
                    </LoadingButton>
                  </Grid>
                </Grid>
              </TabPanel>
            )}

            <TabPanel
              value={value}
              index={taStatus.enabled ? 0 : 1}
              className={styles.commontabPanel}
            >
              <Box>
                <Box className={styles.tabHeading}>
                  <Typography component="p">
                    Secure, Organized Communication via Email.
                  </Typography>
                </Box>
                <Grid container className={styles.settingForm}>
                  <Grid item md={4} lg={4}>
                    <InputLabel>SOC Team Email ID</InputLabel>
                    {userInfo.soc.map((email, index) => (
                      <Box>
                        <span className={styles.dFlex}>
                          <TextField
                            size="small"
                            placeholder="SOC Email ID "
                            id="socMail"
                            value={userInfo.soc[index]}
                            name="soc"
                            error={
                              userInfoError.soc.length
                                ? userInfoError.soc[index].status
                                : ""
                            }
                            helperText={
                              userInfoError.soc.length
                                ? userInfoError.soc[index].message
                                : ""
                            }
                            fullWidth
                            sx={{ mb: "14px" }}
                            onChange={(e) => handleChangeSoc(e, index)}
                          />
                          {index === userInfo.soc.length - 1 ? (
                            userInfo.soc.length == 1 ? (
                              ""
                            ) : (
                              <TfiMinus
                                onClick={() => handleRemoveSoc(index)}
                                className={styles.minusIcon}
                              />
                            )
                          ) : (
                            <TfiMinus
                              onClick={() => handleRemoveSoc(index)}
                              className={styles.minusIcon}
                            />
                          )}{" "}
                        </span>
                        {index === userInfo.soc.length - 1 && (
                          <span
                            className={styles.addButton}
                            onClick={handleAddSoc}
                          >
                            <TfiPlus />
                            Add
                          </span>
                        )}
                      </Box>
                    ))}
                    {userInfo.soc.length == 0 && (
                      <span className={styles.plusIcon} onClick={handleAddSoc}>
                        <TfiPlus />
                        Add
                      </span>
                    )}
                    <LoadingButton
                      disabled={loading.soc}
                      loading={loading.soc}
                      loadingPosition="start"
                      id="socMail"
                      variant="contained"
                      className={styles.btnSuccess}
                      sx={{ mt: "20px" }}
                      onClick={handleClick}
                    >
                      Update
                    </LoadingButton>
                  </Grid>
                </Grid>
              </Box>
            </TabPanel>
            {!taStatus.enabled && (
              <TabPanel
                value={value}
                index={2}
                className={styles.commontabPanel}
              >
                <Box>
                  <Box className={styles.tabHeading}>
                    <Typography component="p">
                      Functionality for users to update their account password
                      for enhanced security.
                    </Typography>
                  </Box>
                  <Grid container className={styles.settingForm}>
                    <Grid item md={4} lg={4}>
                      <InputLabel>Current Password</InputLabel>
                      <TextField
                        size="small"
                        placeholder="Current Password"
                        autoComplete="off"
                        value={userInfo.currentPassword}
                        name="currentPassword"
                        error={userInfoError.currentPassword.status}
                        helperText={userInfoError.currentPassword.message}
                        type={
                          showPassword.currentPassword ? "text" : "password"
                        }
                        fullWidth
                        sx={{ mb: "24px" }}
                        InputProps={{
                          endAdornment: (
                            <InputAdornment position="end">
                              <IconButton
                                onClick={() =>
                                  handleTogglePassword("currentPassword")
                                }
                                edge="end"
                                className={styles.eyeIcon}
                              >
                                {showPassword.currentPassword ? (
                                  <VisibilityOff />
                                ) : (
                                  <Visibility />
                                )}
                              </IconButton>
                            </InputAdornment>
                          ),
                        }}
                        onChange={handleChangeInfo}
                      />
                      <InputLabel>New Password</InputLabel>
                      <TextField
                        size="small"
                        placeholder="New Password"
                        name="newPassword"
                        value={userInfo.newPassword}
                        error={userInfoError.newPassword.status}
                        helperText={userInfoError.newPassword.message}
                        type={showPassword.newPassword ? "text" : "password"}
                        fullWidth
                        sx={{ mb: "24px" }}
                        InputProps={{
                          endAdornment: (
                            <InputAdornment position="end">
                              <IconButton
                                onClick={() =>
                                  handleTogglePassword("newPassword")
                                }
                                edge="end"
                                className={styles.eyeIcon}
                              >
                                {showPassword.newPassword ? (
                                  <VisibilityOff />
                                ) : (
                                  <Visibility />
                                )}
                              </IconButton>
                            </InputAdornment>
                          ),
                        }}
                        onChange={handleChangeInfo}
                      />
                      <InputLabel>Confirm Password</InputLabel>
                      <TextField
                        size="small"
                        placeholder="Confirm Password"
                        name="confirmPassword"
                        type={
                          showPassword.confirmPassword ? "text" : "password"
                        }
                        value={userInfo.confirmPassword}
                        error={userInfoError.confirmPassword.status}
                        helperText={userInfoError.confirmPassword.message}
                        fullWidth
                        sx={{ mb: "24px" }}
                        InputProps={{
                          endAdornment: (
                            <InputAdornment position="end">
                              <IconButton
                                onClick={() =>
                                  handleTogglePassword("confirmPassword")
                                }
                                edge="end"
                                className={styles.eyeIcon}
                              >
                                {showPassword.confirmPassword ? (
                                  <VisibilityOff />
                                ) : (
                                  <Visibility />
                                )}
                              </IconButton>
                            </InputAdornment>
                          ),
                        }}
                        onChange={handleChangeInfo}
                      />
                      <LoadingButton
                        disabled={loading.password}
                        loading={loading.password}
                        loadingPosition="start"
                        id="password"
                        variant="contained"
                        className={styles.btnSuccess}
                        sx={{ mb: "30px" }}
                        onClick={handleClick}
                      >
                        Change Password
                      </LoadingButton>
                    </Grid>
                  </Grid>
                </Box>
              </TabPanel>
            )}

            <TabPanel
              value={value}
              index={taStatus.enabled ? 1 : 3}
              className={styles.commontabPanel}
            >
              <Box>
                <Box className={styles.tabHeading}>
                  <Typography component="p">
                    Enhance communication by notifying users about critical
                    email reports.
                  </Typography>
                </Box>

                <Box className={styles.infoContainer}>
                  <IoInformation className={styles.infoIcon} />
                  The threshold count signifies the minimum number of reporters
                  needed for the admin to broadcast the email to all other users
                  within the organization.
                </Box>
                <Box className={styles.displayFlex}>
                  <Typography variant="h5">Enable Broadcast</Typography>
                  <Switch
                    className={styles.SwitchTab}
                    size="small"
                    color="success"
                    checked={userInfo.broadcastEnabled}
                    onClick={() =>
                      changeBroadcastStatus(userInfo.broadcastEnabled)
                    }
                  />
                </Box>

                {userInfo && userInfo.broadcastEnabled && (
                  <Grid container className={styles.settingForm}>
                    <Grid item md={3} lg={3}>
                      <InputLabel>Threshold Min Limit (Reporters)</InputLabel>
                      <TextField
                        size="small"
                        type="number"
                        placeholder="Enter Threshold"
                        autoComplete="off"
                        value={userInfo.broadcastThreshold}
                        name="broadcastThreshold"
                        error={userInfoError.broadcastThreshold.status}
                        helperText={userInfoError.broadcastThreshold.message}
                        fullWidth
                        sx={{ mb: "24px" }}
                        onChange={handleChangeInfo}
                        inputProps={{
                          inputMode: "numeric",
                          pattern: "[0-9]*",
                          min: 0,
                        }}
                        onKeyDown={(e) => {
                          if (
                            !/[0-9]/.test(e.key) &&
                            e.key !== "Backspace" &&
                            e.key !== "ArrowLeft" &&
                            e.key !== "ArrowRight" &&
                            e.key !== "Delete" &&
                            e.key !== "Tab"
                          ) {
                            e.preventDefault();
                          }
                        }}
                        onInput={(e) => {
                          const value = e.target.value;
                          e.target.value = value.replace(/[^0-9]/g, "");
                          if (e.target.value < 0) {
                            e.target.value = 0;
                          }
                        }}
                      />
                      <LoadingButton
                        disabled={loading.password}
                        loading={loading.password}
                        loadingPosition="start"
                        id="broadcastThreshold"
                        variant="contained"
                        className={styles.btnSuccess}
                        sx={{ mb: "30px" }}
                        onClick={handleClick}
                      >
                        Update
                      </LoadingButton>
                    </Grid>
                  </Grid>
                )}
              </Box>
            </TabPanel>
            <TabPanel
              value={value}
              index={taStatus.enabled ? 2 : 4}
              className={styles.commontabPanel}
            >
              <Box>
                <Box className={styles.tabHeading}>
                  <Typography component="p">
                    Integration for Comprehensive Malware Analysis
                  </Typography>
                </Box>
                <Grid container className={styles.settingForm}>
                  <Grid item md={4} lg={4}>
                    <InputLabel>API Key</InputLabel>
                    <TextField
                      size="small"
                      placeholder="Enter API Key"
                      autoComplete="off"
                      value={userInfo.apiKey}
                      name="apiKey"
                      error={userInfoError.apiKey.status}
                      helperText={userInfoError.apiKey.message}
                      type={showPassword ? "text" : "password"}
                      fullWidth
                      sx={{ mb: "24px" }}
                      onChange={handleChangeInfo}
                    />
                    <LoadingButton
                      disabled={loading.password}
                      loading={loading.password}
                      loadingPosition="start"
                      id="apiKey"
                      variant="contained"
                      className={styles.btnSuccess}
                      sx={{ mb: "30px" }}
                      onClick={handleClick}
                    >
                      Upload
                    </LoadingButton>
                  </Grid>
                </Grid>
              </Box>
            </TabPanel>

            {/* <TabPanel value={value} index={5} className={styles.commontabPanel}>
                <Box>
                    <Box className={styles.tabHeading}>
                      <Typography variant='h6'>License Key</Typography>
                      <Typography component="p">User-specific settings and personal information stored within the system.</Typography>		
                    </Box>
                  <Grid container className={styles.settingForm}> 
                    <Grid item md={4} lg={4}>
                    <InputLabel>License Key</InputLabel> 
                      <TextField 
                        size="small" 
                        placeholder='License Key' 
                        autoComplete='off' 
                        value={userInfo.licenseKey} 
                        name="licenseKey" 
                        error={userInfoError.licenseKey.status} 
                        helperText={userInfoError.licenseKey.message} 
                        fullWidth sx={{ mb: '24px' }} onChange= {handleChangeInfo} />
                      <LoadingButton disabled={loading.password} loading={loading.password} loadingPosition="start" id="licenseKey" variant="contained" className={styles.btnSuccess} sx={{ mb: '30px' }} onClick={handleClick}>Upload</LoadingButton>

                    </Grid>
                  </Grid>
                    
                </Box>
                
            </TabPanel> */}

            <TabPanel
              value={value}
              index={taStatus.enabled ? 3 : 5}
              className={styles.commontabPanel}
            >
              <Box>
                <Box className={styles.tabHeading}>
                  <Box className={styles.displayFlex}>
                    <Typography variant="h6">TSAT Connection</Typography>
                    <Switch
                      className={styles.SwitchTab}
                      size="small"
                      color="success"
                      checked={userInfo.isTsat}
                      onClick={() => changeStatus(userInfo.isTsat)}
                    />
                  </Box>
                  <Typography component="p">Connect TPIR with TSAT</Typography>
                </Box>
                <Grid container className={styles.settingForm}>
                  <Grid item md={4} lg={4}>
                    <InputLabel>TSAT Url</InputLabel>
                    <TextField
                      size="small"
                      placeholder="Enter Url"
                      autoComplete="off"
                      value={userInfo.tsat}
                      name="tsat"
                      error={userInfoError.tsat.status}
                      helperText={userInfoError.tsat.message}
                      // type={showPassword ? "text" : "password"}
                      fullWidth
                      sx={{ mb: "24px" }}
                      onChange={handleChangeInfo}
                    />
                    <LoadingButton
                      disabled={loading.password}
                      loading={loading.password}
                      loadingPosition="start"
                      id="tsat"
                      variant="contained"
                      className={styles.btnSuccess}
                      sx={{ mb: "30px" }}
                      onClick={handleClick}
                    >
                      Upload
                    </LoadingButton>
                  </Grid>
                </Grid>
              </Box>
            </TabPanel>

            <TabPanel
              value={value}
              index={taStatus.enabled ? 4 : 6}
              className={styles.commontabPanel}
            >
              <Box>
                <Grid container className={styles.SmptSection}>
                  <Grid item md={4} lg={4}>
                    <Box className={styles.rightSection}>
                      <Box>
                        <Box className={styles.tabHeading}>
                          <Typography component="p">
                            Settings for Simple Mail Transfer Protocol, enabling
                            the system to send emails.
                          </Typography>
                        </Box>

                        <RadioGroup
                          aria-labelledby="demo-radio-buttons-group-label"
                          value={currentSmtpValue}
                          onChange={handleChangeSmtp}
                          // defaultValue={
                          //   userInfo.smtpHost == null ? "Default" : "Custom"
                          // }
                          name="radio-buttons-group"
                          className={styles.radioBoxBtn}
                        >
                          <FormControlLabel
                            value="Default"
                            control={<Radio Default />}
                            label="Default"
                            onChange={handleCheckboxChange}
                            disabled={smtpId.current}
                          />

                          <FormControlLabel
                            value="Custom"
                            control={<Radio />}
                            label="Custom"
                            onChange={handleCheckboxChangecustom}
                            disabled={smtpId.current}
                          />

                          <FormControlLabel
                            value="Oauth"
                            control={<Radio />}
                            label="Oauth"
                            onChange={handleCheckboxChangecustomOauth}
                          />
                        </RadioGroup>
                      </Box>
                    </Box>

                    {customsmptHideShiw && (
                      <>
                        <Box className={styles.settingForm}>
                          <InputLabel>Username</InputLabel>
                          <TextField
                            size="small"
                            type="text"
                            placeholder="Enter User Name"
                            name="smtpUsername"
                            value={userInfo.smtpUsername}
                            error={userInfoError.smtpUsername.status}
                            helperText={userInfoError.smtpUsername.message}
                            onChange={handleChangeInfo}
                            fullWidth
                            sx={{ mb: "20px" }}
                          />

                          <InputLabel> From</InputLabel>
                          <TextField
                            size="small"
                            type="text"
                            placeholder="Enter From"
                            name="smtpFrom"
                            onChange={handleChangeInfo}
                            error={userInfoError.smtpFrom.status}
                            helperText={userInfoError.smtpFrom.message}
                            value={userInfo.smtpFrom}
                            fullWidth
                            sx={{ mb: "20px" }}
                          />

                          <InputLabel> Host</InputLabel>
                          <TextField
                            size="small"
                            type="text"
                            placeholder="Enter Host"
                            name="smtpHost"
                            error={userInfoError.smtpHost.status}
                            helperText={userInfoError.smtpHost.message}
                            onChange={handleChangeInfo}
                            value={userInfo.smtpHost}
                            fullWidth
                            sx={{ mb: "20px" }}
                          />

                          <InputLabel> Password</InputLabel>
                          <TextField
                            size="small"
                            type="password"
                            placeholder="Enter Password"
                            name="smtpPassword"
                            error={userInfoError.smtpPassword.status}
                            helperText={userInfoError.smtpPassword.message}
                            onChange={handleChangeInfo}
                            value={userInfo.smtpPassword}
                            fullWidth
                            sx={{ mb: "20px" }}
                          />

                          <InputLabel> Port</InputLabel>
                          <TextField
                            size="small"
                            type="text"
                            placeholder="Enter Port"
                            name="smtpPort"
                            error={userInfoError.smtpPort.status}
                            helperText={userInfoError.smtpPort.message}
                            onChange={handleChangeInfo}
                            value={userInfo.smtpPort}
                            fullWidth
                            sx={{ mb: "24px" }}
                          />

                          <Box className={styles.settingUpdateBtn}>
                            <LoadingButton
                              disabled={loading.info}
                              loading={loading.info}
                              loadingPosition="start"
                              id="customSmtp"
                              onClick={handleClick}
                              variant="contained"
                              className={styles.btnSuccess}
                              sx={{ mb: "10px", mr: "5px" }}
                            >
                              Update
                            </LoadingButton>
                            <LoadingButton
                              disabled={loading.info}
                              loading={loading.info}
                              loadingPosition="start"
                              id="smtpModal"
                              variant="contained"
                              onClick={handleClick}
                              className={styles.btnSuccess}
                              sx={{ mb: "10px" }}
                            >
                              Send Test Email
                            </LoadingButton>
                            <LoadingButton
                              className={styles.btnDelete}
                              id={"clearCustomSmtp"}
                              disabled={loading.info}
                              loading={loading.info}
                              loadingPosition="start"
                              variant="contained"
                              sx={{ ml: "5px", mb: "10px" }}
                              onClick={handleClick}
                            >
                              clear
                            </LoadingButton>

                          </Box>
                        </Box>
                      </>
                    )}

                    {showCustomOauth && (
                      <>
                        <FormControl>
                          <RadioGroup
                            row
                            value={selectedmicroSoftValue}
                            onChange={handleChangemicroSoft}
                            name="radio-buttons"
                            sx={{ mt: 2 }}
                          >
                            <Box sx={{ display: "flex", gap: 2, mb: 2 }}>
                              <Button
                                className={styles.oauthToggleBtn}
                                variant={
                                  selectedmicroSoftValue === "Direct"
                                    ? "contained"
                                    : "outlined"
                                }
                                color={
                                  selectedmicroSoftValue === "Direct"
                                    ? "primary"
                                    : "inherit"
                                }
                                onClick={() =>
                                  handleChangemicroSoft({
                                    target: { value: "Direct" },
                                  })
                                }
                                startIcon={
                                  <span
                                    className={`${styles.oauthToggleCircle} ${
                                      selectedmicroSoftValue === "Direct"
                                        ? styles.active
                                        : ""
                                    }`}
                                  >
                                    {selectedmicroSoftValue === "Direct" && (
                                      <IoCheckmark
                                        style={{
                                          color: "#1976d2",
                                          fontSize: 12,
                                          verticalAlign: "middle",
                                        }}
                                      />
                                    )}
                                  </span>
                                }
                                disabled={
                                  smtpId.current == "manual" ? true : false
                                }
                              >
                                Direct
                              </Button>
                              <Button
                                className={styles.oauthToggleBtn}
                                variant={
                                  selectedmicroSoftValue === "Manual"
                                    ? "contained"
                                    : "outlined"
                                }
                                color={
                                  selectedmicroSoftValue === "Manual"
                                    ? "primary"
                                    : "inherit"
                                }
                                onClick={() =>
                                  handleChangemicroSoft({
                                    target: { value: "Manual" },
                                  })
                                }
                                startIcon={
                                  <span
                                    className={`${styles.oauthToggleCircle} ${
                                      selectedmicroSoftValue === "Direct"
                                        ? styles.active
                                        : ""
                                    }`}
                                  >
                                    {selectedmicroSoftValue === "Manual" && (
                                      <IoCheckmark
                                        style={{
                                          color: "#1976d2",
                                          fontSize: 12,
                                          verticalAlign: "middle",
                                        }}
                                      />
                                    )}
                                  </span>
                                }
                                disabled={
                                  smtpId.current == "direct" ? true : false
                                }
                              >
                                Manual
                              </Button>
                            </Box>
                          </RadioGroup>
                        </FormControl>

                        {selectedmicroSoftValue === "Manual" && (
                          <Box className={styles.commontabPanel}>
                            <Grid container className={styles.settingForm}>
                              <InputLabel>Client ID</InputLabel>
                              <TextField
                                size="small"
                                type="text"
                                placeholder="Enter Client ID"
                                name="clientId"
                                onChange={handleChangeInfo}
                                fullWidth
                                disabled={smtpId.showClear}
                                error={userInfoError.clientId.status}
                                helperText={userInfoError.clientId.message}
                                value={userInfo.clientId}
                                sx={{ mb: "20px" }}
                              />

                              <InputLabel>Secret Key</InputLabel>
                              <TextField
                                size="small"
                                type="text"
                                placeholder="Enter Secret Key"
                                name="secretKey"
                                disabled={smtpId.showClear}
                                onChange={handleChangeInfo}
                                error={userInfoError.secretKey.status}
                                helperText={userInfoError.secretKey.message}
                                value={userInfo.secretKey}
                                fullWidth
                                sx={{ mb: "20px" }}
                              />

                              <InputLabel>Tenant ID</InputLabel>
                              <TextField
                                size="small"
                                type="text"
                                placeholder="Enter Tenant ID"
                                name="tenantId"
                                disabled={smtpId.showClear}
                                onChange={handleChangeInfo}
                                error={userInfoError.tenantId.status}
                                helperText={userInfoError.tenantId.message}
                                value={userInfo.tenantId}
                                fullWidth
                                sx={{ mb: "20px" }}
                              />

                              <InputLabel>From</InputLabel>
                              <TextField
                                size="small"
                                placeholder="Email ID "
                                value={userInfo.senderEmail}
                                disabled={smtpId.showClear}
                                name="senderEmail"
                                fullWidth
                                sx={{ mb: "20px" }}
                                onChange={handleChangeInfo}
                              />
                              {!smtpId.showClear && (
                                <LoadingButton
                                  className={styles.btnSuccess}
                                  id={"oauthManualSave"}
                                  disabled={loading.info}
                                  loading={loading.info}
                                  loadingPosition="start"
                                  variant="contained"
                                  sx={{ mr: "10px", mt: "10px" }}
                                  onClick={handleClick}
                                >
                                  Save
                                </LoadingButton>
                              )}
                              {smtpId.showClear && (
                                <LoadingButton
                                  className={styles.btnSuccess}
                                  id={"sendMailOauth"}
                                  disabled={loading.info}
                                  loading={loading.info}
                                  loadingPosition="start"
                                  variant="contained"
                                  sx={{ mr: "10px", mt: "10px" }}
                                  onClick={smtphandleOpen}
                                >
                                  Send Test Email
                                </LoadingButton>
                              )}
                              {smtpId.showClear && (
                                <LoadingButton
                                  className={styles.btnDelete}
                                  id={"clearOauth"}
                                  disabled={loading.info}
                                  loading={loading.info}
                                  loadingPosition="start"
                                  variant="contained"
                                  sx={{ mr: "10px", mt: "10px" }}
                                  onClick={handleClick}
                                >
                                  clear
                                </LoadingButton>
                              )}

                              {/* <LoadingButton
                  className={styles.btnSuccess}
                  id={"user"}
                  disabled={loading.info}
                  loading={loading.info}
                  loadingPosition="start"
                  variant="contained"
                  sx={{ mt: "10px", mr: "10px" }}
                  
                  >
                  Advanced Settings
                  </LoadingButton> */}
                            </Grid>
                          </Box>
                        )}

                        {selectedmicroSoftValue === "Direct" && (
                          <Grid container className={styles.settingForm}>
                            <InputLabel>From</InputLabel>
                            <TextField
                              size="small"
                              type="email"
                              placeholder="Enter Email Address"
                              name="oauthDirectEmailForm"
                              disabled={smtpId.showClear}
                              onChange={handleChangeInfo}
                              error={userInfoError.oauthDirectEmailForm.status}
                              helperText={
                                userInfoError.oauthDirectEmailForm.status &&
                                userInfoError.oauthDirectEmailForm.message
                              }
                              value={userInfo.oauthDirectEmailForm}
                              fullWidth
                              sx={{ mb: "20px" }}
                            />
                            <Typography
                              variant="h3"
                              sx={{
                                fontStyle: "italic",
                                fontSize: "13px",
                                color: "#000",
                                lineHeight: "18px",
                              }}
                            >
                              <b>*Note:</b> The email address must exist and be
                              linked to a valid Azure mailbox. It will be used
                              as the sender when emailing users.
                            </Typography>
                            <Stack
                              direction="row"
                              spacing={1}
                              sx={{ mt: "20px" }}
                            >
                              {!smtpId.showClear && (
                                <LoadingButton
                                  disabled={loading.info}
                                  loading={loading.info}
                                  loadingPosition="start"
                                  variant="contained"
                                  id="directOauth"
                                  onClick={handleClick}
                                  className={styles.btnSuccess}
                                  sx={{ mb: "10px", mt: "20px" }}
                                >
                                  Authorize
                                </LoadingButton>
                              )}
                              {smtpId.showClear && (
                                <LoadingButton
                                  disabled={loading.info}
                                  loading={loading.info}
                                  loadingPosition="start"
                                  variant="contained"
                                  id="sendMailOauth"
                                  onClick={smtphandleOpen}
                                  className={styles.btnSuccess}
                                  sx={{ mb: "10px", mt: "20px" }}
                                >
                                  Send Test Email
                                </LoadingButton>
                              )}
                              {smtpId.showClear && (
                                <LoadingButton
                                  disabled={loading.info}
                                  loading={loading.info}
                                  loadingPosition="start"
                                  variant="contained"
                                  id="clearOauth"
                                  onClick={handleClick}
                                  className={styles.btnDelete}
                                  sx={{ mb: "10px", mt: "20px" }}
                                >
                                  Clear
                                </LoadingButton>
                              )}
                            </Stack>
                          </Grid>
                        )}
                      </>
                    )}

                    {defultsmptHideShiw && currentSmtpValue === "Default" && (
                      <Box className={styles.settingForm}>
                        <LoadingButton
                          disabled={loading.info}
                          loading={loading.info}
                          loadingPosition="start"
                          variant="contained"
                          id="defaultSmtp"
                          onClick={handleClick}
                          className={styles.btnSuccess}
                          sx={{ mb: "10px" }}
                        >
                          Update
                        </LoadingButton>
                      </Box>
                    )}
                  </Grid>
                </Grid>
              </Box>
            </TabPanel>

            <TabPanel
              value={value}
              index={taStatus.enabled ? 5 : 7}
              className={styles.commontabPanel}
            >
              <Box className={styles.settingPanel}>
                <Grid container className={styles.SmptSection}>
                  <Box className={styles.tabHeading}>
                    <Typography component="p">
                      Tailor email formats to suit specific needs.
                    </Typography>
                  </Box>
                  <Grid item md={12}>
                    <Box className={styles.templatesSection}>
                      <TableContainer className={styles.tableContainer}>
                        <Table
                          sx={{ minWidth: 650 }}
                          aria-label="simple table"
                          className={styles.ReportedMailTable}
                        >
                          <TableHead className={styles.thead_background}>
                            <TableRow>
                              <TableCell>Template</TableCell>
                              <TableCell>Description</TableCell>
                              <TableCell>Action</TableCell>
                            </TableRow>
                          </TableHead>
                          {templates && templates.length > 0 && (
                            <TableBody className={styles.reportedMail}>
                              {templates.map((data, index) => {
                                return (
                                  <TableRow
                                    key={index}
                                    sx={{
                                      "&:last-child td, &:last-child th": {
                                        border: 0,
                                      },
                                    }}
                                  >
                                    <TableCell component="td" scope="row">
                                      <b>{data.name}</b>
                                    </TableCell>
                                    <TableCell component="td" scope="row">
                                      {data.description}
                                    </TableCell>
                                    <TableCell component="td" scope="row">
                                      <Link href={`/template/${data.name}`}>
                                        <TiEdit className={styles.EditBtn} />
                                      </Link>
                                    </TableCell>
                                  </TableRow>
                                );
                              })}
                            </TableBody>
                          )}
                        </Table>
                      </TableContainer>
                    </Box>
                  </Grid>
                </Grid>
              </Box>
            </TabPanel>
            <TabPanel
              value={value}
              index={taStatus.enabled ? 6 : 8}
              className={styles.commontabPanel}
            >
              <Box>
                <Box className={styles.tabHeading}>
                  {/* <Box className={styles.displayFlex}>
                        <Typography variant='h6'>Time Zone</Typography>	
                      </Box> */}
                  <Typography component="p">Select Time Zone</Typography>
                </Box>
                <Grid container className={styles.settingForm}>
                  <Grid item md={4} lg={4}>
                    <FormControl fullWidth>
                      <label>Select Timezone</label>
                      <Select
                        labelId="demo-simple-select-label"
                        id="demo-simple-select"
                        name="timeZone"
                        value={timeZone}
                        onChange={(e) => setTimeZone(e.target.value)}
                        displayEmpty
                        size="small"
                        className="selectTimeZone"
                        sx={{ marginBottom: "14px", fontSize: "14px" }}
                      >
                        <MenuItem value="" disabled>
                          Select Timezone
                        </MenuItem>
                        {timezones.map((timezone, index) => (
                          <MenuItem
                            key={index}
                            value={timezone}
                            sx={{ fontSize: "14px", marginTop: "0.5rem" }}
                          >
                            {timezone}
                          </MenuItem>
                        ))}
                      </Select>
                    </FormControl>
                    <LoadingButton
                      disabled={loading.password}
                      loading={loading.password}
                      loadingPosition="start"
                      id="tsat"
                      variant="contained"
                      className={styles.btnSuccess}
                      sx={{ mb: "30px" }}
                      onClick={updateTimezone}
                    >
                      Set
                    </LoadingButton>
                  </Grid>
                </Grid>
              </Box>
            </TabPanel>

            <TabPanel value={value} index={9} className={styles.commontabPanel}>
              <Box className={styles.settingPanel}>
                <Grid container className={styles.SmptSection}>
                  <Box className={styles.tabHeading}>
                    <Typography component="p">
                      Manage your security settings and 2FA preferences.
                    </Typography>
                  </Box>
                  <Grid item md={12}>
                    <Box className={styles.twoFactorBtn}>
                      <Typography component="p">
                        Two Factor Authentication
                      </Typography>

                      <Button
                        variant="contained"
                        className={styles.btn_new_project}
                        onClick={handleOpen}
                      >
                        {" "}
                        Two Factor Authentication{" "}
                      </Button>
                    </Box>
                  </Grid>
                </Grid>
              </Box>
            </TabPanel>

            {!taStatus.isNewCompany && (
              <TabPanel
                value={value}
                index={taStatus.enabled ? 7 : 10}
                className={styles.commontabPanel}
              >
                <Box className={styles.settingPanel}>
                  <Grid container className={styles.SmptSection}>
                    <Box className={styles.tabHeading}>
                      <Typography component="p">
                        Easily manage your connection between Threatcop Admin
                        and your current product.
                      </Typography>
                    </Box>
                    <Grid item md={12}>
                      {!taStatus.enabled && !taStatus.connecting && (
                        <Button
                          variant="contained"
                          className={`${styles.btn} ${styles.btnPrimary}`}
                          onClick={taConnecthandleOpen}
                        >
                          <TfiLink style={{ marginRight: "5px" }} />
                          Start Integration
                        </Button>
                      )}
                      {taStatus.enabled && !taStatus.connecting && (
                        <Button
                          variant="contained"
                          className={`${styles.btn} ${styles.btnPrimary}`}
                          onClick={disconnectSwal}
                        >
                          <TfiLink style={{ marginRight: "5px" }} />
                          Disable Integration
                        </Button>
                      )}
                      {!taStatus.enabled && taStatus.connecting && (
                        <Button
                          variant="contained"
                          disabled
                          className={`${styles.btn} ${styles.btnPrimary}`}
                        >
                          <TfiLink style={{ marginRight: "5px" }} />
                          Connection with TA in progess
                        </Button>
                      )}
                    </Grid>
                  </Grid>
                </Box>
              </TabPanel>
            )}

            <TabPanel
              value={value}
              index={(!taStatus.isNewCompany && taStatus.enabled) ? 8 : taStatus.enabled &&taStatus.isNewCompany ? 7 : 11}
              className={styles.commontabPanel}
            >
              <Box>
                <Grid container className={styles.SmptSection}>
                  <Grid item md={8}>
                    <Box className={styles.rightSection}>
                      <Box>
                        <Box className={styles.tabHeading}>
                          <Typography component="p">
                            Choose your preferred reporting method for
                            employees.
                          </Typography>
                        </Box>

                        <RadioGroup
                          aria-labelledby="demo-radio-buttons-group-label"
                          defaultValue={
                            userInfo.reportingMethod == "Default"
                              ? "Default"
                              : "Email Client"
                          }
                          name="radio-buttons-group"
                          className={styles.radioBoxBtn}
                        >
                          <FormControlLabel
                            value="Default"
                            control={<Radio />}
                            onChange={() => updateReportingMethod("default")}
                            label="Default"
                          />
                          <FormControlLabel
                            value="Email Client"
                            control={<Radio Default />}
                            onChange={() =>
                              updateReportingMethod("emailClient")
                            }
                            label="Email Client"
                          />
                        </RadioGroup>
                      </Box>
                    </Box>

                    {userInfo && userInfo.reportingMethod == "Email Client" && (
                      <Box className={styles.contentbarWrap}>
                        <TableContainer
                          className={`${styles.tableContainer} ${styles.domainsTable}`}
                        >
                          <Table aria-label="simple table">
                            <TableHead className={styles.thead_background}>
                              <TableRow>
                                <TableCell>Domain</TableCell>
                                <TableCell>Mailbox</TableCell>
                                <TableCell
                                  className={`${styles.ThSix} ${styles.thCenter}`}
                                >
                                  Credentials
                                </TableCell>

                                <TableCell>Email</TableCell>
                              </TableRow>
                            </TableHead>

                            <TableBody>
                              {domains &&
                                domains.length &&
                                domains.map((domain) => (
                                  <TableRow
                                    sx={{
                                      "&:last-child td, &:last-child th": {
                                        border: 0,
                                      },
                                    }}
                                  >
                                    <TableCell scope="row">
                                      {domain.domainName}
                                    </TableCell>
                                    <TableCell className={styles.tdCap}>
                                      {domain?.domainType}
                                    </TableCell>

                                    <TableCell sx={{ textAlign: "left" }}>
                                      <span
                                        className={
                                          domain?.isCredential
                                            ? styles.LinkSafe
                                            : styles.LinkDanger
                                        }
                                      >
                                        {domain?.isCredential
                                          ? "Uploaded"
                                          : "Not Uploaded"}
                                      </span>
                                    </TableCell>
                                    <TableCell className={styles.selectOptions}>
                                      {!domain.webhookEmail && (
                                        <span
                                          onClick={() => {
                                            if (!domain.isCredential) {
                                              Swal.fire({
                                                icon: "warning",
                                                title: "Missing Credentials",
                                                text: "Kindly upload credentials before configuring email.",
                                                confirmButtonText:
                                                  "Go to Domains",
                                                confirmButtonColor: "#1967d2",
                                              }).then((result) => {
                                                if (result.isConfirmed) {
                                                  window.location.href =
                                                    "/domains"; // Update with actual navigation method
                                                }
                                              });
                                            } else {
                                              handleOpenConfigure(
                                                domain._id,
                                                domain.webhookEmail
                                              );
                                            }
                                          }}
                                          style={{
                                            textDecoration: "underline",
                                            color: "#1967d2",
                                            cursor: "pointer",
                                          }}
                                        >
                                          Configure Email
                                        </span>
                                      )}
                                      {domain?.isCredential &&
                                        domain?.webhookEmail?.length && (
                                          <span
                                            style={{
                                              color: "#2d993c",
                                              marginLeft: "10px",
                                            }}
                                          >
                                            Configured{" "}
                                            <span
                                              style={{
                                                color: "#1967d2",
                                                cursor: "pointer",
                                              }}
                                              onClick={() =>
                                                handleOpenConfigure(
                                                  domain._id,
                                                  domain.webhookEmail
                                                )
                                              }
                                            >
                                              <TiEdit />
                                            </span>
                                          </span>
                                        )}
                                    </TableCell>
                                  </TableRow>
                                ))}
                            </TableBody>
                          </Table>
                        </TableContainer>
                      </Box>
                    )}
                  </Grid>
                </Grid>
              </Box>
            </TabPanel>
          </Box>
        </Box>

        <Modal
          open={openModal}
          onClose={handleClose}
          aria-labelledby="add domain"
          aria-describedby="add domain"
          className={styles.mainModalBox}
        >
          <Box className={styles.modalBox}>
            <Box className={styles.modalHeader}>
              <Typography id="modal-modal-title" variant="h6" component="h2">
                Two Factor Authentication Type
              </Typography>
              <CloseSharpIcon
                className={styles.closeIcon}
                onClick={handleClose}
              />
            </Box>
            <Box className={styles.commonpaddingDomainType}>
              <Typography component="p">
                Select Two- Factor Authentication Type
              </Typography>
              <Box className={styles.domain_type}>
                <Box
                  className={
                    getAuthType.mail
                      ? styles.active_domain_box
                      : styles.common_domainBox
                  }
                  onClick={() => enable2FA("MAIL")}
                >
                  <SiGmail className={styles.SelectDomainIcon} />
                  <Typography
                    variant="h6"
                    style={{ fontSize: "1rem" }}
                    className={getAuthType.mail ? styles.color_domain_type : ""}
                  >
                    Mail Authentication{" "}
                    {getAuthType.mail ? (
                      <IoCheckmarkOutline style={{ marginLeft: "5px" }} />
                    ) : (
                      ""
                    )}
                  </Typography>
                </Box>

                <Box
                  className={
                    getAuthType.totp
                      ? styles.active_domain_box
                      : styles.common_domainBox
                  }
                  onClick={() => enable2FA("TOTP")}
                >
                  <TfiMobile className={styles.SelectDomainIcon} />
                  <Typography
                    variant="h6"
                    style={{ fontSize: "1rem" }}
                    className={getAuthType.totp ? styles.color_domain_type : ""}
                  >
                    {" "}
                    Authenticator App{" "}
                    {getAuthType.totp ? (
                      <IoCheckmarkOutline style={{ marginLeft: "5px" }} />
                    ) : (
                      ""
                    )}{" "}
                  </Typography>
                </Box>
              </Box>

              <Box className={styles.twoFactorBtn}>
                {getAuthType.is2fa &&
                  disableState == "mail" &&
                  getAuthType.mail == true && (
                    <Box className={styles.domain_enable_btn}>
                      {" "}
                      <Button
                        variant="contained"
                        color="success"
                        onClick={() => disable2FA()}
                        className={styles.btndisable}
                      >
                        Disable
                      </Button>{" "}
                    </Box>
                  )}
                {getAuthType.is2fa &&
                  disableState == "totp" &&
                  getAuthType.totp == true && (
                    <Box className={styles.domain_enable_btn}>
                      {" "}
                      <Button
                        variant="contained"
                        color="success"
                        onClick={() => disable2FA()}
                        className={styles.btndisable}
                      >
                        Disable
                      </Button>{" "}
                    </Box>
                  )}
                <Box className={styles.domain_enable_btn}>
                  <Button
                    variant="contained"
                    color="success"
                    onClick={() => set2FA()}
                    className={styles.btnEnable}
                  >
                    Enable
                  </Button>
                </Box>
              </Box>
            </Box>
          </Box>
        </Modal>

        <Modal
          open={smptopenModal}
          onClose={smpthandleClose}
          aria-labelledby="add domain"
          aria-describedby="add domain"
          className={styles.mainModalBox}
        >
          <Box className={`${styles.modalBox} ${styles.smptModal}`}>
            <Box className={styles.modalHeader}>
              <Typography id="modal-modal-title" variant="h6" component="h2">
                Send Test Mail
              </Typography>
              <CloseSharpIcon
                className={styles.closeIcon}
                onClick={smpthandleClose}
              />
            </Box>

            <Box className={styles.smptModalBox}>
              <Typography component="p">Enter Test E-Mail Id</Typography>
              <Box className={styles.textFoem}>
                <TextField
                  size="small"
                  type="email"
                  placeholder="Enter Email"
                  name="testMail"
                  onChange={handleChangeInfo}
                  value={userInfo.testMail}
                  helperText={userInfoError.testMail.message}
                  error={userInfoError.testMail.status}
                  sx={{ width: "100%", mb: "20px" }}
                />
              </Box>
              <LoadingButton
                variant="contained"
                loading={loading.testMail}
                className={styles.primaryBtn}
                id="sendTestMail"
                onClick={handleClick}
              >
                Send
              </LoadingButton>
            </Box>
          </Box>
        </Modal>

        <Modal
          open={openConfigure}
          onClose={handleConfigureClose}
          aria-labelledby="modal-title"
          aria-describedby="modal-description"
          className={styles.ConfigureModalBox}
        >
          <Box className={styles.modalBox}>
            {/* Modal Content Based on Selected Action */}
            <Box className={styles.modalHeader}>
              <Box>
                <Typography variant="h6">Email Configuration</Typography>
                <Typography component="p">
                  Set your email for the specific domain.
                </Typography>
              </Box>

              <CloseSharpIcon
                className={styles.closeIcon}
                onClick={handleConfigureClose}
              />
            </Box>

            <Box className={styles.modalBody}>
              <InputLabel>Email Address</InputLabel>
              <TextField
                size="small"
                type="text"
                placeholder="Enter Email"
                error={userInfoError?.webhookEmail?.status}
                helperText={userInfoError?.webhookEmail?.message}
                value={userInfo?.webhookEmail}
                onChange={handleChangeInfo}
                name="webhookEmail"
                fullWidth
                sx={{ mb: "20px" }}
              />

              <Alert className={styles.alert} severity="info">
                To begin using the client reporting method, you must configure
                the email ID for the domain.
              </Alert>
            </Box>

            <Box className={styles.modalFooter}>
              <Grid container>
                <Grid item xs={12} md={12} align="right" position="relative">
                  <Button
                    variant="contained"
                    id="webhookEmail"
                    onClick={handleClick}
                  >
                    Add
                  </Button>
                </Grid>
              </Grid>
            </Box>
          </Box>
        </Modal>
      </Box>
      <NewconnecttaModal
        taConnecthandleClose={taConnecthandleClose}
        taConnectopen={taConnectopen}
        loadingTaConnection={loadingTaConnection}
        changeConnectionStatusWithTa={changeConnectionStatusWithTa}
      />
      {/* <ConnectTaModal
        taConnectopen={taConnectopen}
        changeConnectionStatusWithTa={changeConnectionStatusWithTa}
        loadingTaConnection={loadingTaConnection}
        taConnecthandleClose={taConnecthandleClose}
      /> */}
    </Box>
  );
}
